/* eslint-disable no-template-curly-in-string */
import { ExtendedFieldConfig, getFieldLabels, isEmpty, ReadonlyField } from '@yandex-infracloud-ui/libs';
import { TextField2, InputField2 } from '@yandex-infracloud-ui/libs-next';
import { FormikProps } from 'formik';
import { config } from 'services';
import { number, object, Schema, string, mixed } from 'yup';
import {
   DefaultDeployNetwork,
   DefaultProvisioner,
   DeployNetwork,
   deployTagSchema,
   IConstants,
   IDeployConfig,
   INewProject,
   ProjectType,
   Provisioner,
} from '../../../models';
import {
   BotProjectField,
   AutomationPlotField,
   MaintenancePlotField,
   DeployConfigField,
   DeployTagsField,
   InputWithCounterField,
   EnumField,
   TypeField,
} from '../../../rich_shared/form_fields2';

export type NewProjectShortFormParams = {
   automation_plot?: string;
   bot_project_id: number | null;
   deployConfig: IDeployConfig | null;
   deploy_network: DeployNetwork | null;
   deploy_tags: Set<string>;
   id: string;
   maintenance_plot_id: string;
   name: string;
   provisioner: Provisioner;
   reason: string;
   yc_iam_folder_id?: string;
   type?: ProjectType;
};

export const initialValue: NewProjectShortFormParams = {
   automation_plot: '',
   bot_project_id: null,
   deployConfig: {
      config: '',
      provisioner: DefaultProvisioner,
   },
   deploy_network: DefaultDeployNetwork,
   deploy_tags: new Set<string>(),
   id: '',
   maintenance_plot_id: '',
   name: '',
   provisioner: DefaultProvisioner,
   reason: '',
   yc_iam_folder_id: undefined,
};

// TODO избавиться от INewProject
export const formParamsToNewProject = ({ deployConfig, ...params }: NewProjectShortFormParams): INewProject => {
   const project: INewProject = {
      automation_plot_id: params.automation_plot || undefined,
      maintenance_plot_id: params.maintenance_plot_id || undefined,
      bot_project_id: params.bot_project_id!,
      deploy_config: deployConfig!.config,
      id: params.id,
      name: params.name,
      provisioner: deployConfig!.provisioner,
      reason: params.reason,
      yc_iam_folder_id: params.yc_iam_folder_id || undefined,
      type: params.type,
   };

   if (deployConfig!.provisioner === Provisioner.LUI) {
      project.deploy_network = params.deploy_network!;
   } else {
      project.deploy_tags = params.deploy_tags;
   }

   return project;
};

const hasDeployConfig = (name: string, f: FormikProps<NewProjectShortFormParams>) =>
   f.values.deployConfig && !isEmpty(f.values.deployConfig.config);

const hasNotDeployConfig = (name: string, f: FormikProps<NewProjectShortFormParams>) => !hasDeployConfig(name, f);

const fieldMaintenancePlot: ExtendedFieldConfig<NewProjectShortFormParams> = {
   as: MaintenancePlotField,
   label: 'Maintenance plot',
   name: 'maintenance_plot_id',
};
const fieldFolderId: ExtendedFieldConfig<NewProjectShortFormParams> = {
   as: InputField2,
   label: 'Folder ID',
   name: 'yc_iam_folder_id',
   required: true,
   placeholder: 'Enter a folder ID for YC iam',
};

export const fields: ExtendedFieldConfig<NewProjectShortFormParams>[] = [
   {
      as: TypeField,
      label: 'Type',
      name: 'type',
   },
   {
      as: InputWithCounterField,
      hint: 'Human readable ID',
      label: 'Project name',
      name: 'name',
      placeholder: 'Enter project name',
      required: true,
   },
   {
      as: InputWithCounterField,
      hint: 'Tech ID',
      label: 'Project ID/slug',
      name: 'id',
      placeholder: 'Enter project ID/slug',
      required: true,
   },
   {
      as: BotProjectField,
      label: 'BOT Project ID',
      name: 'bot_project_id',
      placeholder: 'Enter ABC service',
      required: true,
   },
   {
      as: DeployConfigField,
      label: 'Deploy config',
      name: 'deployConfig',
      placeholder: 'Enter deploy config',
      required: true,
   },
   // @ts-ignore
   ...(config.isExternal ? [fieldFolderId] : []),
   {
      as: ReadonlyField,
      controlProps: {
         valueFormat: (v: string) => v.toUpperCase(),
      },
      hidden: hasNotDeployConfig,
      label: 'Provisioner',
      name: 'provisioner',
      readonly: true,
   },
   {
      as: EnumField,
      controlProps: {
         options: [
            { value: DeployNetwork.Project, content: 'Project' },
            { value: DeployNetwork.Service, content: 'Service' },
         ],
      },
      help: "Deploy host either in service or in project VLAN's",
      // deploy_network – часть настроек наливки, надо показывать только если выбран provisioner=lui
      hidden: (_, f) => hasNotDeployConfig(_, f) || f.values.provisioner !== Provisioner.LUI,
      label: 'Network',
      name: 'deploy_network',
   },
   {
      as: DeployTagsField,
      help: 'Einstellung tags to apply to host before deployment',
      // deploy_tags – часть настроек наливки, надо показывать только если выбран provisioner=eine
      hidden: (_, f) => hasNotDeployConfig(_, f) || f.values.provisioner !== Provisioner.EINE,
      label: 'Deploy tags',
      name: 'deploy_tags',
      placeholder: 'Enter deploy tags',
   },
   {
      as: AutomationPlotField,
      label: 'Automation plot',
      name: 'automation_plot',
   },
   ...(!config.isExternal ? [fieldMaintenancePlot] : []),
   {
      as: TextField2,
      hint: 'Will be logged',
      label: 'Comment',
      name: 'reason',
      placeholder: 'Enter a reason for project creation',
   },
];

const getLabel = getFieldLabels(fields);

function checkProvisioner(value: Provisioner) {
   // @ts-ignore
   const constants: IConstants = this.options.context.constants;
   if (!constants) {
      // throw new Error('Need constants for validation');
      return true; // TODO
   }

   return constants.provisioners.includes(value);
}

const provisionerSchema = string()
   .label('Provisioner')
   .required()
   // eslint-disable-next-line
   .test('provisioner', '${path} must be one of supported provisioners', checkProvisioner) as Schema<Provisioner>;

const validationShape = {
   maintenance_plot_id: string().label(getLabel('maintenance_plot_id')!),
   bot_project_id: number()
      .label(getLabel('bot_project_id')!)
      .nullable()
      .required('${path} is a required field')
      .typeError('${path} must be an integer')
      .integer('${path} must be a positive integer')
      .positive('${path} must be a positive integer'),

   deployConfig: object<IDeployConfig>({
      config: string().label(getLabel('deployConfig')!).required(),
      provisioner: provisionerSchema,
   })
      .nullable()
      .required(),
   deploy_network: string().when('provisioner', {
      is: Provisioner.LUI,
      otherwise: string(),
      then: string()
         .required()
         .oneOf([DeployNetwork.Project, DeployNetwork.Service] as string[]),
   }) as Schema<DeployNetwork>,
   deploy_tags: deployTagSchema,
   id: string().label(getLabel('id')!).required(),
   name: string().label(getLabel('name')!).required(),
   provisioner: provisionerSchema,
   reason: string().label(getLabel('reason')!),
   type: mixed<ProjectType>().oneOf(Object.values(ProjectType)),
};

export const validationSchemaInternal = object<NewProjectShortFormParams>({
   ...validationShape,
   automation_plot: string().label(getLabel('automation_plot')!),
});

export const validationSchemaExternal = object<NewProjectShortFormParams>({
   ...validationShape,
   yc_iam_folder_id: string().label(getLabel('yc_iam_folder_id')!).required(),
});
