import { DISMISS_REASON, IValidationResult, modalService, toasts, useDismounted } from '@yandex-infracloud-ui/libs';
import * as React from 'react';
import { useCallback, useEffect, useMemo, useState } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import { StaticContext } from 'react-router';
import { RouteComponentProps } from 'react-router-dom';
import { finalize, takeUntil } from 'rxjs/operators';

import { IProject, ProjectType, ROUTE_LINKS } from '../../models';
import { ProjectReportSettingsForm } from '../../rich_shared';
import { ProjectAutomationTumblers } from '../../rich_shared/project_automations';
import { config, projectApi } from '../../services';
import {
   DevJson,
   InnerScrollPage,
   ProjectIdContext,
   ReasonModal,
   ScrollSection,
   SideScrollMenu,
   useConstants,
   useWalleTitle,
} from '../../shared';
import {
   addOwner,
   clear,
   isProjectFormValid,
   loadAutomations,
   loadProject,
   reset,
   setAutomationLimits,
   setAutomations,
   setBasicFields,
   setDisabledChecks,
   setPlots,
   setCms,
   addCms,
   setDeploying,
   setNewProject,
   setNotifications,
   setOthers,
   setProfiling,
   setReports,
   setRestrictions,
   setVlans,
   setCAuthFields,
   setHostLimits,
} from '../../state/fullProjectForm';
import { addProject } from '../../state/projects';
import { RootState } from '../../state/store';
import { CloneProjectModal } from './CloneProjectModal/CloneProjectModal';
import { ProjectHeader } from './components/ProjectHeader';
import { formParamsToProject } from './models';
import classes from './ProjectPage.module.css';
import { RemoveProjectModal } from './RemoveProjectModal/RemoveProjectModal';

import { AutomationsForm } from './sub_forms/AutomationsForm/AutomationsForm';
import { BasicForm } from './sub_forms/BasicForm/BasicForm';
import { CAuthForm } from './sub_forms/CAuthForm/CAuthForm';
import { DisabledChecksForm } from './sub_forms/DisabledChecksForm/DisabledChecksForm';
import { PlotsForm } from './sub_forms/PlotsForm/PlotsForm';
import { CmsForm } from './sub_forms/CmsForm/CmsForm';
import { DeployingForm } from './sub_forms/DeployingForm/DeployingForm';
import { HostLimits } from './sub_forms/HostLimits/HostLimits';
import { NotificationsForm } from './sub_forms/NotificationsForm/NotificationsForm';
import { OthersForm } from './sub_forms/OthersForm/OthersForm';
import { ProfilingForm } from './sub_forms/ProfilingForm/ProfilingForm';
import { ProjectAutomationLimits } from './sub_forms/ProjectAutomationLimits/ProjectAutomationLimits';
import { RestrictionsForm } from './sub_forms/RestrictionsForm/RestrictionsForm';
import { VlansForm } from './sub_forms/VlansForm/VlansForm';

function getError<T>(vr: IValidationResult<T> | undefined) {
   return vr === undefined || vr.isValid ? '' : 'Errors in this block';
}

export const ProjectPage: React.FC<RouteComponentProps<{ id: string }, StaticContext, { prev: any }>> = React.memo(
   ({ location, history, match }) => {
      //region hooks
      const [readonly, setReadonly] = useState(true);
      const [isSaving, setSaving] = useState(false);
      const [isOwnersSaving, setOwnersSaving] = useState(false);
      const fullProjectForm = useSelector((s: RootState) => s.fullProjectForm);

      const {
         allAvailableProjectChecks,
         automationLimits,
         automations,
         basic,
         cauth,
         plots,
         cmsSettings,
         deploying,
         dnsZone,
         hostLimits,
         isNewProject,
         disabledChecks,
         notifications,
         others,
         profiling,
         reports,
         restrictions,
         vlans,
      } = fullProjectForm;

      const dismounted = useDismounted();
      const dispatch = useDispatch();

      const user = useSelector((s: RootState) => s.globals.user);
      const isShadow = basic.params.type === ProjectType.SHADOW;
      const canEdit = useMemo(
         () =>
            (config.isExternal || (user ? user.isAdmin || user.projects.includes(basic.params.id) : false)) &&
            !isShadow,
         [basic.params.id, user, isShadow],
      );
      const { constants } = useConstants();

      const isLoaded = isNewProject !== null;
      //endregion

      //region effects
      // Текущего пользователя по-умолчанию добавляем во владельцы
      useEffect(() => {
         if (isLoaded && isNewProject && user !== null && !config.isExternal && !isShadow) {
            dispatch(addOwner(user.login));
         } else {
            // при типе external надо явно это сделать, иначе валидация пропустит пустую форму
            dispatch(setBasicFields(basic.params));
         }
         // basic.params не статичные и меняются от setBasicFields, если мы их в deps добавим, то уйдет в луп
         // eslint-disable-next-line react-hooks/exhaustive-deps
      }, [isLoaded, isNewProject, user, dispatch]);

      useEffect(() => {
         if (match.params.id === 'new') {
            setReadonly(false);
            dispatch(setNewProject());
         } else {
            dispatch(loadProject(match.params.id, dismounted));
         }
      }, [dismounted, dispatch, match]);

      // Сброс состояния после ухода со страницы
      useEffect(
         () => () => {
            dispatch(clear());
         },
         [dispatch],
      );

      useWalleTitle(isNewProject === null ? 'Project loading' : isNewProject ? 'New project' : basic.params.name);
      //endregion

      //region handlers
      const onBasicChange = useCallback(v => dispatch(setBasicFields(v)), [dispatch]);

      const onCAuthChange = useCallback(v => dispatch(setCAuthFields(v)), [dispatch]);

      const onCmsChange = useCallback(
         (v, params = null, index = null) => dispatch(setCms(v, params, index, constants!)),
         [dispatch, constants],
      );

      const onCmsAdd = useCallback(() => dispatch(addCms(constants!)), [constants, dispatch]);

      const onDisabledChecksChange = useCallback(v => dispatch(setDisabledChecks(v)), [dispatch]);

      const onVlansChange = useCallback(v => dispatch(setVlans(v)), [dispatch]);

      const onDeployingChange = useCallback(v => dispatch(setDeploying(v, constants!)), [constants, dispatch]);

      const onProfilingChange = useCallback(v => dispatch(setProfiling(v)), [dispatch]);

      const onAutomationLimitsChange = useCallback(v => dispatch(setAutomationLimits(v)), [dispatch]);

      const onHostLimitsChange = useCallback(v => dispatch(setHostLimits(v)), [dispatch]);

      const onPlotsChange = useCallback(v => dispatch(setPlots(v)), [dispatch]);

      const onRestrictionsChange = useCallback(v => dispatch(setRestrictions(v)), [dispatch]);

      const onNotificationsChange = useCallback(v => dispatch(setNotifications(v)), [dispatch]);

      const onReportsChange = useCallback(v => dispatch(setReports(v)), [dispatch]);

      const onAutomationsChange = useCallback(v => dispatch(setAutomations(v)), [dispatch]);

      const onOthersChange = useCallback(v => dispatch(setOthers(v)), [dispatch]);

      const handleClone = useCallback(() => {
         modalService
            .open(CloneProjectModal, { initialProjectId: isNewProject ? undefined : basic.params.id })
            .pipe(takeUntil(dismounted))
            .subscribe(
               project => dispatch(addProject(project as any)),
               reason => {
                  if (reason !== DISMISS_REASON) {
                     toasts.apiError('Cloning project', reason);
                  }
               },
            );
      }, [basic.params.id, dismounted, dispatch, isNewProject]);

      const goBack = useCallback(() => {
         const backUrl = location.state && location.state.prev ? location.state.prev : ROUTE_LINKS.hostList();
         history.push(backUrl);
      }, [history, location.state]);

      const handleCancel = useCallback(() => {
         if (isNewProject) {
            goBack();

            return;
         }

         setReadonly(true);
         dispatch(reset());
      }, [dispatch, goBack, isNewProject]);

      const handleRemove = useCallback(() => {
         modalService
            .open(RemoveProjectModal, { projectId: basic.params.id })
            .pipe(takeUntil(dismounted))
            .subscribe(
               () => history.push(ROUTE_LINKS.hostList()),
               reason => {
                  if (reason !== DISMISS_REASON) {
                     toasts.apiError('Project removing', reason);
                  }
               },
            );
      }, [basic.params.id, dismounted, history]);

      const handleSave = useCallback(() => {
         if (!isProjectFormValid(fullProjectForm)) {
            toasts.error('Please, fix form errors', 'Project saving');

            return;
         }

         modalService
            .open(ReasonModal, { reason: 'Reason' })
            .pipe(takeUntil(dismounted))
            .subscribe(
               ({ reason = '' }) => {
                  const actualProject = formParamsToProject(fullProjectForm);

                  if (isNewProject) {
                     projectApi
                        .create({ ...actualProject, reason } as any)
                        .pipe(takeUntil(dismounted))
                        .subscribe(p => {
                           dispatch(addProject(p as any));
                           setReadonly(true);
                           history.push(ROUTE_LINKS.project(p.id));
                           toasts.success(`Project ${p.name} has been created`);
                        }, toasts.handleApiError('Project creating'));
                  } else {
                     const originalProject = formParamsToProject(fullProjectForm.originalProject);
                     setSaving(true);

                     const [projectSavingObs, ownersSavingObs] = projectApi.save(
                        originalProject as any,
                        actualProject as IProject,
                        reason,
                     );

                     projectSavingObs
                        .pipe(
                           finalize(() => setSaving(false)),
                           takeUntil(dismounted),
                        )
                        .subscribe(() => {
                           dispatch(loadProject(basic.params.id, dismounted));
                           setReadonly(true);
                        }, toasts.handleApiError('Project saving'));

                     if (ownersSavingObs !== null) {
                        setOwnersSaving(true);

                        ownersSavingObs
                           .pipe(
                              finalize(() => setOwnersSaving(false)),
                              takeUntil(dismounted),
                           )
                           .subscribe(
                              () => dispatch(loadProject(basic.params.id, dismounted)),
                              toasts.handleApiError('Project owners saving'),
                           );
                     }
                  }
               },
               () => null,
            );
      }, [basic.params.id, dismounted, dispatch, fullProjectForm, history, isNewProject]);

      const handleAutomationNeedUpdate = useCallback(() => dispatch(loadAutomations(match.params.id, dismounted)), [
         dismounted,
         dispatch,
         match.params.id,
      ]);

      const setEditMode = useCallback(() => setReadonly(false), []);
      //endregion

      //region render
      const header = (
         <ProjectHeader
            isNew={isNewProject}
            isSaving={isSaving}
            readonly={readonly}
            canEdit={canEdit}
            projectId={basic.params.id}
            projectName={basic.params.name}
            onEdit={setEditMode}
            onCancel={handleCancel}
            onSave={handleSave}
            onRemove={handleRemove}
            onClone={handleClone}
         />
      );

      if (isNewProject === null) {
         return <InnerScrollPage header={header} />;
      }

      const automationSection = (
         <ScrollSection id={'automations'} title={'Automations'} error={getError(automations.validation)}>
            <div className={classes.form}>
               {isNewProject ? (
                  <AutomationsForm
                     value={automations.params}
                     validation={automations.validation}
                     onChange={onAutomationsChange}
                  />
               ) : (
                  <ProjectAutomationTumblers
                     disabled={!canEdit}
                     dnsDomain={automations.params.dns_domain}
                     dnsZone={dnsZone.params}
                     dnsAutomation={automations.params.dns_automation}
                     fsmEnabled={automations.params.fsm}
                     fsmDetails={automations.params.fsmDetails}
                     healingAutomation={automations.params.healing_automation}
                     projectId={basic.params.id}
                     projectName={basic.params.name}
                     onChange={handleAutomationNeedUpdate}
                  />
               )}
            </div>
            <DevJson>{automations}</DevJson>
         </ScrollSection>
      );

      const basicSection = (
         <ScrollSection id={'basic'} title={'Basic settings'} error={getError(basic.validation)}>
            <div className={classes.form}>
               <ProjectIdContext.Provider value={basic.params.id}>
                  <BasicForm
                     isNew={isNewProject}
                     projectId={isNewProject ? undefined : basic.params.id}
                     isOwnersSaving={isOwnersSaving}
                     readonly={readonly || isSaving}
                     value={basic.params}
                     validation={basic.validation}
                     onChange={onBasicChange}
                     isShadow={isShadow}
                  />
               </ProjectIdContext.Provider>
            </div>
            <DevJson>{basic}</DevJson>
         </ScrollSection>
      );

      const cauthSection = (
         <ScrollSection id={'cauth'} title={'CAuth'} error={getError(cauth.validation)}>
            <div className={classes.form}>
               <CAuthForm
                  readonly={readonly || isSaving}
                  value={cauth.params}
                  onChange={onCAuthChange}
                  validation={cauth.validation}
               />
            </div>
            <DevJson>{cauth}</DevJson>
         </ScrollSection>
      );

      const cmsSection = (
         <ScrollSection
            id={'cms'}
            title={'Cluster Management System'}
            error={getError(cmsSettings.find(e => !e.validation.isValid)?.validation)}
         >
            <div className={classes.form}>
               <CmsForm readonly={readonly || isSaving} value={cmsSettings} onChange={onCmsChange} onAdd={onCmsAdd} />
            </div>
            <DevJson>{cmsSettings}</DevJson>
         </ScrollSection>
      );

      const vlansSection = (
         <ScrollSection id={'vlans'} title={'VLANs'} error={getError(vlans.validation)}>
            <div className={classes.form}>
               <VlansForm
                  readonly={readonly || isSaving}
                  value={vlans.params}
                  onChange={onVlansChange}
                  validation={vlans.validation}
               />
            </div>
            <DevJson>{vlans}</DevJson>
         </ScrollSection>
      );

      const deployingSection = (
         <ScrollSection id={'deploying'} title={'Deploying'} error={getError(deploying.validation)}>
            <div className={classes.form}>
               <DeployingForm
                  readonly={readonly || isSaving}
                  value={deploying.params}
                  validation={deploying.validation}
                  onChange={onDeployingChange}
               />
            </div>
            <DevJson>{deploying}</DevJson>
         </ScrollSection>
      );

      const profilingSection = (
         <ScrollSection id={'profiling'} title={'Profiling'} error={getError(profiling.validation)}>
            <div className={classes.form}>
               <ProfilingForm
                  readonly={readonly || isSaving}
                  value={profiling.params}
                  validation={profiling.validation}
                  onChange={onProfilingChange}
               />
            </div>
            <DevJson>{profiling}</DevJson>
         </ScrollSection>
      );

      const automationLimitsSection = (
         <ScrollSection
            id={'automation_limits'}
            title={'Automation limits'}
            error={getError(automationLimits.validation)}
         >
            <div className={classes.automationLimits}>
               <ProjectAutomationLimits
                  readonly={readonly || isSaving}
                  value={automationLimits.params}
                  validation={automationLimits.validation}
                  onChange={onAutomationLimitsChange}
               />
            </div>
            <DevJson>{automationLimits}</DevJson>
         </ScrollSection>
      );

      const hostLimitsSection = (
         <ScrollSection id={'host_limits'} title={'Host limits'} error={getError(hostLimits.validation)}>
            <div className={classes.hostLimits}>
               <HostLimits
                  readonly={readonly || isSaving}
                  value={hostLimits.params}
                  validation={hostLimits.validation}
                  onChange={onHostLimitsChange}
               />
            </div>
            <DevJson>{hostLimits}</DevJson>
         </ScrollSection>
      );

      const plotsSection = (
         <ScrollSection id={'plots'} title={'Plots'} error={getError(plots.validation)}>
            <div className={classes.automationLimits}>
               <PlotsForm
                  readonly={readonly || isSaving}
                  value={plots.params}
                  onChange={onPlotsChange}
                  validation={plots.validation}
                  isShadow={isShadow}
               />
            </div>
            <DevJson>{plots}</DevJson>
         </ScrollSection>
      );

      const restrictionSection = (
         <ScrollSection id={'restrictions'} title={'Host restrictions'} error={getError(restrictions.validation)}>
            <p className={classes.subHeader}>
               These setting are applied to new hosts only and have no impact to active project ones.
            </p>

            <div className={classes.form}>
               <RestrictionsForm
                  readonly={readonly || isSaving}
                  value={restrictions.params}
                  validation={restrictions.validation}
                  onChange={onRestrictionsChange}
               />
            </div>
            <DevJson>{restrictions}</DevJson>
         </ScrollSection>
      );
      const notificationsSection = (
         <ScrollSection id={'notifications'} title={'Notifications'} error={getError(notifications.validation)}>
            <div className={classes.form}>
               <NotificationsForm
                  readonly={readonly || isSaving}
                  value={notifications.params}
                  validation={notifications.validation}
                  onChange={onNotificationsChange}
               />
            </div>
            <DevJson>{notifications}</DevJson>
         </ScrollSection>
      );
      const disabledChecksSection = (
         <ScrollSection id={'disabledChecks'} title={'Disabled checks'} error={getError(disabledChecks.validation)}>
            <div className={classes.form}>
               <DisabledChecksForm
                  readonly={readonly || isSaving}
                  value={disabledChecks.params}
                  allAvailableProjectChecks={allAvailableProjectChecks.params}
                  validation={disabledChecks.validation}
                  onChange={onDisabledChecksChange}
               />
            </div>
            <DevJson>{disabledChecks}</DevJson>
         </ScrollSection>
      );
      const reportsSection = (
         <ScrollSection id={'reports'} title={'Reports'} error={getError(reports.validation)}>
            <div className={classes.form}>
               <ProjectReportSettingsForm
                  readonly={readonly || isSaving}
                  value={reports.params}
                  validation={reports.validation}
                  onChange={onReportsChange}
               />
            </div>
            <DevJson>{reports}</DevJson>
         </ScrollSection>
      );

      const othersSection = (
         <ScrollSection id={'others'} title={'Others'} error={getError(others.validation)}>
            <div className={classes.form}>
               <OthersForm
                  readonly={readonly || isSaving}
                  value={others.params}
                  validation={others.validation}
                  onChange={onOthersChange}
                  isNew={isNewProject}
               />
            </div>
            <DevJson>{others}</DevJson>
         </ScrollSection>
      );

      // noinspection PointlessBooleanExpressionJS
      return (
         <InnerScrollPage header={header}>
            <SideScrollMenu>
               {isShadow === false && isNewProject === false ? automationSection : null}

               {basicSection}

               {isShadow === false ? cauthSection : null}
               {isShadow === false ? cmsSection : null}
               {isShadow === false ? vlansSection : null}
               {isShadow === false ? deployingSection : null}
               {isShadow === false ? profilingSection : null}
               {isShadow === false ? automationLimitsSection : null}
               {isShadow === false ? hostLimitsSection : null}

               {isShadow === true && config.isExternal === true ? null : plotsSection}

               {isShadow === false ? restrictionSection : null}
               {isShadow === false && isNewProject === false ? disabledChecksSection : null}
               {isShadow === false ? notificationsSection : null}
               {isShadow === false ? reportsSection : null}
               {isShadow === false && isNewProject ? automationSection : null}
               {isShadow === false ? othersSection : null}
            </SideScrollMenu>
         </InnerScrollPage>
      );
      //endregion
   },
);

ProjectPage.displayName = 'ProjectPage';
