import {
   classNames,
   DISMISS_REASON,
   FormHooks,
   FormHooksContext,
   IModalProps,
   Loader,
   ModalLayout,
   parseApiError,
   useDismounted,
} from '@yandex-infracloud-ui/libs';
import { Form, Formik, FormikHelpers } from 'formik';
import React, { PropsWithChildren, useCallback, useMemo, useState } from 'react';
import { useDispatch } from 'react-redux';
import { takeUntil } from 'rxjs/operators';
import { object, string } from 'yup';

import { IApiError, IAutomationForm, IProjectActionParams } from '../../../models';
import { ReasonField } from '../../../rich_shared/form_fields2/ReasonField';
import { projectApi } from '../../../services';
import { LegoButton } from '../../../shared/lego';
import { projectsSlice } from '../../../state/projects';
import classes from './RemoveProjectModal.module.css';

const initialValue: IProjectActionParams = {
   reason: '',
};

const validationSchema = object<IProjectActionParams>({
   reason: string().label('Comment'),
});

const InternalForm: React.FC<PropsWithChildren<{}>> = ({ children }) => (
   <Form className={classes.form}>
      <ReasonField />

      {children}
   </Form>
);

interface Props extends IModalProps<void> {
   projectId: string;
}

export const RemoveProjectModal: React.FC<Props> = ({ projectId, ok, cancel }) => {
   const dismounted = useDismounted();
   const [apiError, setApiError] = useState('');
   const dispatch = useDispatch();

   const formHooks = useMemo(
      () =>
         ({
            onFieldChange: () => setApiError(''),
         } as FormHooks<IAutomationForm>),
      [],
   );

   const handleSubmit = useCallback(
      (v: IProjectActionParams, helpers: FormikHelpers<IProjectActionParams>) => {
         projectApi
            .remove(projectId, v)
            .pipe(takeUntil(dismounted))
            .subscribe(
               () => {
                  dispatch(projectsSlice.actions.removeProject(projectId));
                  ok();
                  helpers.setSubmitting(false);
               },
               (resp: IApiError) => {
                  setApiError(parseApiError(resp));
                  helpers.setSubmitting(false);
               },
            );
      },
      [dismounted, dispatch, ok, projectId],
   );

   const dismiss = useCallback(() => cancel(DISMISS_REASON), [cancel]);

   return (
      <ModalLayout onDismiss={dismiss} title={'Remove project'} showFooter={false}>
         <Formik initialValues={initialValue} onSubmit={handleSubmit} validationSchema={validationSchema}>
            {form => (
               <FormHooksContext.Provider value={formHooks}>
                  <InternalForm>
                     <footer className={classes.footer}>
                        {apiError && !form.isSubmitting ? <span className={classes.error}>{apiError}</span> : null}

                        <Loader visible={form.isSubmitting} inline={true} text={'Project is removing'} />

                        <LegoButton theme={'clear'} cls={classes.button} onClick={dismiss} disabled={form.isSubmitting}>
                           Cancel
                        </LegoButton>

                        <LegoButton
                           theme={'action'}
                           type={'submit'}
                           cls={classNames(classes.button, classes.danderButton)}
                           disabled={!form.isValid || form.isSubmitting || form.isValidating}
                        >
                           Remove project
                        </LegoButton>
                     </footer>
                  </InternalForm>
               </FormHooksContext.Provider>
            )}
         </Formik>
      </ModalLayout>
   );
};

RemoveProjectModal.displayName = 'RemoveProjectModal';
