import {
   FormFieldType,
   FormLayout,
   IEnumOption,
   IFormField,
   isEmpty,
   isEqual,
   replaceItem,
} from '@yandex-infracloud-ui/libs';
import * as React from 'react';
import { useCallback, useEffect, useState } from 'react';

import { useConstants } from '../../../../shared';
import { BooleanSwitcherField, EnumField } from '../../../../rich_shared/form_fields';
import { CmsFormParams, CmsType } from '../../../../state/fullProjectForm';

import { FormProps } from '../../models';
import { CmsApiUrlField } from './CmsApiUrlField';
import { useSelector } from 'react-redux';
import { RootState } from 'state/store';

export const cmsFields = (isTvmRequired: boolean = true): IFormField<CmsFormParams>[] => [
   {
      label: 'CMS type',
      name: '_type',
      options: [
         { title: 'Default', value: CmsType.Default },
         { title: 'Custom', value: CmsType.Custom },
      ],
      type: FormFieldType.Custom,
      component: EnumField,
   },
   {
      isHidden: p => p._type !== CmsType.Default,
      label: 'Max № simultaneously handled hosts',
      name: 'max_busy_hosts',
      type: FormFieldType.Number,
   },
   {
      isHidden: p => p._type !== CmsType.Custom,
      label: 'API version',
      name: 'api_version',
      options: [], // loaded from constants in CmsForm
      required: true,
      type: FormFieldType.Custom,
      component: EnumField,
   },
   {
      label: 'Temporary unreachable',
      name: 'temporary_unreachable_enabled',
      type: FormFieldType.Custom,
      component: BooleanSwitcherField,
      isHidden: p => p._type !== CmsType.Custom || Number(p?.api_version?.slice(1)) < 1.4,
   },
   {
      /*
      TVM-app применим только для custom cms, для default и YP эта настройка неприменима.
      TVM-app является обязательным полем при изменении настроек cms.
      */
      isHidden: p => !(p._type === CmsType.Custom),
      label: 'TVM App ID',
      name: 'tvm_app_id',
      placeholder: 'Enter TVM Application ID',
      required: isTvmRequired,
      type: FormFieldType.Number,
   },
   {
      component: CmsApiUrlField,
      isHidden: p => p._type !== CmsType.Custom,
      label: 'API URL',
      name: 'url',
      placeholder: 'Enter URL for custom CMS API',
      required: true,
      type: FormFieldType.Custom,
   },
];

interface Props extends Omit<FormProps<CmsFormParams>, 'onChange'> {
   onChange(v: CmsFormParams, index: number): void;
   index: number;
   readonly: boolean;
}

export const CmsFormItem: React.FC<Props> = React.memo(({ readonly, index, value, onChange, validation }) => {
   const user = useSelector((s: RootState) => s.globals.user);
   const isAdmin = user ? user.isAdmin : false;

   const [fields, setFields] = useState(cmsFields(!isAdmin));
   const { constants } = useConstants();

   // Проверка новых версий с бекенда
   useEffect(() => {
      if (constants === null) {
         return;
      }

      const apiField = fields.find(f => f.name === 'api_version')!;
      const versions = constants.cms_api_versions
         .filter(v => v !== 'YP') // TODO remove when backend is ready (YP - особый случай)
         .map(v => ({ value: v, title: v } as IEnumOption));

      if (!isEqual(apiField.options, versions)) {
         setFields(pf =>
            replaceItem(
               pf,
               f => f === apiField,
               f => ({ ...f, options: versions }),
            ),
         );
      }
   }, [constants, fields]);

   useEffect(() => {
      onChange({ ...value, _isTvmRequired: !isAdmin }, index);
      // eslint-disable-next-line react-hooks/exhaustive-deps
   }, []);

   const onFormUpdate = useCallback(
      (_, v: CmsFormParams) => {
         const apiField = fields.find(f => f.name === 'api_version')!;

         // _type changed
         if (!isEqual(value._type, v._type)) {
            switch (v._type) {
               case CmsType.Default: {
                  v.url = 'default';
                  v.temporary_unreachable_enabled = undefined;
                  break;
               }
               case CmsType.Custom: {
                  if (!v.api_version) {
                     v.api_version = apiField.options![0].value;
                  }
                  if (Number(v.api_version.slice(1)) < 1.4) {
                     v.temporary_unreachable_enabled = false;
                  }
                  v.url = '';
                  break;
               }
            }
         }

         // api_version changed
         if (!isEqual(value.api_version, v.api_version)) {
            if (Number(v?.api_version?.slice(1)) < 1.4) {
               v.temporary_unreachable_enabled = undefined;
            } else {
               v.temporary_unreachable_enabled = false;
            }
         }

         // tvm_app_id changed
         if (!isEqual(value.tvm_app_id, v.tvm_app_id)) {
            if (isEmpty(v.tvm_app_id)) {
               v.tvm_app_id = undefined;
            }
         }

         onChange(v, index);
      },
      [onChange, value, index, fields],
   );

   return (
      <FormLayout
         readonly={readonly}
         fields={fields}
         value={value}
         validationResult={validation}
         onChange={onFormUpdate}
      />
   );
});

CmsFormItem.displayName = 'CmsFormItem';
