import { FormFieldType, FormLayout, IFormField, isEmpty, toasts } from '@yandex-infracloud-ui/libs';
import * as React from 'react';
import { useCallback, useEffect, useState } from 'react';

import { AutomationLimit, IAutomationPlot } from '../../../../models';
import { AutomationPlotSelectField, MaintenancePlotSelectField } from '../../../../rich_shared/form_fields';
import { automationPlotsApi, config } from '../../../../services';
import { PlotsFormParams } from '../../../../state/fullProjectForm';
import { FormProps } from '../../models';

import { Automation } from '../ProjectAutomationLimits/Automation';
import classes from './PlotsForm.module.css';

const fieldMaintenance: IFormField<PlotsFormParams> = {
   component: MaintenancePlotSelectField,
   label: 'Maintenance plot',
   name: 'maintenance_plot_id',
   placeholder: 'Select maintenance plot',
   type: FormFieldType.Custom,
   isHidden: () => config.isExternal,
};

const fieldAutomation: IFormField<PlotsFormParams> = {
   component: AutomationPlotSelectField,
   label: 'Automation plot',
   name: 'automation_plot_id',
   placeholder: 'Select automation plot',
   type: FormFieldType.Custom,
};

interface Props extends FormProps<PlotsFormParams> {
   isShadow: boolean;
   readonly: boolean;
}

export const PlotsForm = React.memo(({ readonly, value, onChange, validation, isShadow }: Props) => {
   const [plot, setPlot] = useState<IAutomationPlot | null>(null);

   useEffect(() => {
      if (!value.automation_plot_id) {
         setPlot(p => (p ? null : p));

         return;
      }

      const s = automationPlotsApi
         .getById(value.automation_plot_id)
         .subscribe(setPlot, toasts.handleApiError('Automation plot loading'));

      return () => s.unsubscribe();
   }, [onChange, value.automation_plot_id]);

   const onFormUpdate = useCallback(
      (_, v: PlotsFormParams) => {
         onChange({
            ...value,
            automation_plot_id: v.automation_plot_id,
            maintenance_plot_id: v.maintenance_plot_id,
         });
      },
      [value, onChange],
   );

   const onAutomationUpdate = useCallback(
      (name: string, limits: AutomationLimit[]) => {
         onChange({
            ...value,
            automation_limits: {
               ...value.automation_limits,
               [name]: limits,
            },
         });
      },
      [value, onChange],
   );

   return (
      <>
         <FormLayout
            readonly={readonly}
            fields={
               config.isExternal
                  ? [fieldAutomation]
                  : isShadow
                  ? [fieldMaintenance]
                  : [fieldMaintenance, fieldAutomation]
            }
            value={value}
            onChange={onFormUpdate}
            validationResult={validation}
         />
         {!isShadow && plot ? (
            <>
               <h4>Maximum count of failed checks</h4>
               {isEmpty(plot.checks)
                  ? `There are not checks in automation plot "${plot.name}"`
                  : plot.checks.map(check => {
                       const limitKey = `max_${check.name}_failures`;

                       return (
                          <Automation
                             className={classes.automation}
                             key={check.name}
                             name={limitKey}
                             readonly={readonly}
                             value={value.automation_limits![limitKey] || []}
                             onChange={onAutomationUpdate}
                          />
                       );
                    })}
            </>
         ) : null}
      </>
   );
});

PlotsForm.displayName = 'PlotsForm';
