import { autobind, classNames } from '@yandex-infracloud-ui/libs';
import { Button, Icon, Popup, Select, TextInput } from 'lego-on-react';
import * as React from 'react';

import deleteSvg from '../../../../design/icons/delete.svg';
import { AutomationLimit, HELP_HINTS } from '../../../../models';

import classes from './AutomationRow.module.css';
import { CustomAutomationPopup } from './CustomAutomationPopup';
import addSvg from './icons/add.svg';
import editSvg from './icons/edit.svg';

interface Props {
   className?: string;
   first: boolean;
   index: number;
   name: string;
   readonly: boolean;
   value: AutomationLimit;

   onAdd(): void;

   onChange(index: number, v: AutomationLimit): void;

   onDelete(index: number): void;
}

interface IState {
   customFormat: boolean;
   popupOpened: boolean;
}

// TODO портировать на FC
export class AutomationRow extends React.PureComponent<Props, IState> {
   public static defaultProps = {
      className: '',
   };

   private _hardcodedPeriods = [
      { period: '10m', name: '10 minutes' },
      { period: '1h', name: 'hour' },
      { period: '2h', name: '2 hours' },
      { period: '1d', name: 'day' },
   ];

   private _nameRef = React.createRef<HTMLDivElement>();

   constructor(props: Props) {
      super(props);

      this.state = {
         customFormat: this._hasCustomFormat(this.props.value),
         popupOpened: false,
      };
   }

   public componentWillReceiveProps(nextProps: Readonly<Props>): void {
      this.setState({
         customFormat: this._hasCustomFormat(nextProps.value),
         popupOpened: false,
      });
   }

   public render() {
      const componentClasses = classNames(
         {
            [classes.hovered]: this.state.popupOpened,
         },
         classes.automationRow,
         this.props.className,
      );
      const hint =
         this.props.name in HELP_HINTS.automationLimits
            ? `${this.props.name}: ${HELP_HINTS.automationLimits[this.props.name]}`
            : this.props.name;

      return (
         <div className={componentClasses}>
            <div className={classes.name} ref={this._nameRef}>
               {this.props.first ? (
                  <span title={hint}>{formatAutomationLimitName(this.props.name)}</span>
               ) : (
                  <span className={classes.and}>and</span>
               )}
            </div>

            <div className={classes.value}>{this._renderValueControl()}</div>

            {this.props.readonly ? null : (
               <div className={classes.buttons}>
                  {this.state.customFormat ? (
                     <Button
                        theme={'link'}
                        size={'s'}
                        text={'Edit'}
                        onClick={this._openPopup}
                        icon={<Icon url={editSvg} />}
                     />
                  ) : null}

                  {this.props.first ? (
                     <Button
                        theme={'link'}
                        size={'s'}
                        name={`${this.props.name}_add-button`}
                        text={'Add condition'}
                        onClick={this.props.onAdd}
                        icon={<Icon url={addSvg} />}
                     />
                  ) : (
                     <Button
                        theme={'link'}
                        size={'s'}
                        cls={classes.deleteButton}
                        name={`${this.props.name}_delete-button`}
                        text={'Delete'}
                        onClick={() => this.props.onDelete(this.props.index)}
                        icon={<Icon url={deleteSvg} />}
                     />
                  )}
               </div>
            )}

            <Popup
               theme={'normal'}
               target={'anchor'}
               hasTail={true}
               directions={['right-center']}
               onOutsideClick={this._closePopup}
               visible={this.state.popupOpened}
               anchor={this._nameRef.current}
            >
               <CustomAutomationPopup
                  hint={HELP_HINTS.automationLimits[this.props.name] || ''}
                  name={this.props.name}
                  value={this.props.value}
                  onSubmit={this._onChange}
                  onCancel={this._closePopup}
               />
            </Popup>
         </div>
      );
   }

   @autobind
   private _closePopup() {
      this.setState({
         popupOpened: false,
      });
   }

   private _hasCustomFormat(value: AutomationLimit) {
      return !this._hardcodedPeriods.some(p => p.period === value.getFullPeriod());
   }

   @autobind
   private _onChange(_: any, value: AutomationLimit): void {
      this.props.onChange(this.props.index, value);
      this._closePopup();
   }

   @autobind
   private _onPeriodChange(values: string | string[]) {
      const [fullPeriod] = values as string[];
      if (fullPeriod === 'other') {
         this._openPopup();

         return;
      }

      let value: AutomationLimit;

      try {
         value = this.props.value.updateFullPeriod(fullPeriod);
      } catch (e) {
         // Невалидное значение пропускаем
         return;
      }

      this.props.onChange(this.props.index, value);
   }

   @autobind
   private _onTimesChange(rawValue: string) {
      const times = parseInt(rawValue, 10);

      let value: AutomationLimit;
      try {
         value = this.props.value.updateTimes(times);
      } catch (e) {
         // Невалидное значение пропускаем
         return;
      }

      this.props.onChange(this.props.index, value);
   }

   @autobind
   private _openPopup() {
      this.setState({
         popupOpened: true,
      });
   }

   private _renderValueControl() {
      const { value } = this.props;

      if (this.state.customFormat || this.props.readonly) {
         return value.toString();
      }

      return (
         <>
            <TextInput
               theme={'normal'}
               size={'s'}
               type={'number'}
               cls={classes.times}
               name={this.props.name}
               text={value.times.toString()}
               onChange={this._onTimesChange}
            />

            <div className={classes.preposition}>{value.preposition()}</div>

            {/* TODO заменить Select на Dropdown из ui-components */}
            <div data-e2e={'AutomationRow:' + this.props.name}>
               <Select
                  theme={'normal'}
                  size={'s'}
                  type={'radio'}
                  val={this.state.customFormat ? 'other' : value.getFullPeriod()}
                  onChange={this._onPeriodChange}
               >
                  {this._hardcodedPeriods.map(p => (
                     <Select.Item key={p.period} val={p.period}>
                        {p.name}
                     </Select.Item>
                  ))}
                  <Select.Item val={'other'}>other...</Select.Item>
               </Select>
            </div>
         </>
      );
   }
}

export function formatAutomationLimitName(name: string): string {
   return name.replace(/^max_/, '').replace(/_failures$/, '');
}
