import { autobind } from '@yandex-infracloud-ui/libs';
import { Button, Select, TextInput } from 'lego-on-react';
import { func, object, string } from 'prop-types';
import * as React from 'react';
import { SyntheticEvent } from 'react';

import { AutomationLimit, PERIOD_LIST, ShortPeriod } from '../../../../models';

import classes from './CustomAutomationPopup.module.css';

interface IProps {
   hint: string;
   value: AutomationLimit;
   name?: string;

   onSubmit(e: SyntheticEvent, v: AutomationLimit): void;

   onCancel(e: SyntheticEvent): void;
}

interface IState {
   value: AutomationLimit;
}

export class CustomAutomationPopup extends React.PureComponent<IProps, IState> {
   public static defaultProps = {
      hint: '',
   };

   public static propTypes = {
      hint: string,
      onCancel: func.isRequired,
      onSubmit: func.isRequired,
      value: object.isRequired,
   };

   constructor(props: IProps) {
      super(props);

      this.state = {
         value: this.props.value,
      };
   }

   public render() {
      const { value } = this.state;

      return (
         <form className={classes.customAutomationPopup} onSubmit={this._onSubmit}>
            <h3>Custom limit</h3>

            {this.props.hint ? <p>{this.props.hint}</p> : null}

            <div className={classes.value}>
               <TextInput
                  theme={'normal'}
                  size={'s'}
                  type={'number'}
                  cls={classes.times}
                  name={`${this.props.name}_time`}
                  text={value.times.toString()}
                  onChange={this._onTimesChange}
               />

               <div className={classes.preposition}>{value.prepositionWithTimes(true)}</div>

               <TextInput
                  theme={'normal'}
                  size={'s'}
                  type={'number'}
                  cls={classes.times}
                  name={`${this.props.name}_period`}
                  text={value.periodValue.toString()}
                  onChange={this._onPeriodValueChange}
               />

               {/* TODO заменить Select на Dropdown из ui-components */}
               <div data-e2e={'CustomAutomationPopup:' + this.props.name}>
                  <Select theme={'normal'} size={'s'} type={'radio'} val={value.period} onChange={this._onPeriodChange}>
                     {PERIOD_LIST.map(p => (
                        <Select.Item key={p.letter} val={p.letter}>
                           {value.periodValue === 1 ? p.one : p.several}
                        </Select.Item>
                     ))}
                  </Select>
               </div>
            </div>

            <div className={classes.footer}>
               <Button theme={'pseudo'} size={'s'} text={'Cancel'} onClick={this.props.onCancel} />
               <Button
                  theme={'action'}
                  size={'s'}
                  text={'Done'}
                  name={`${this.props.name}_done-button`}
                  onClick={this._onSubmit}
               />
            </div>
         </form>
      );
   }

   @autobind
   private _onPeriodChange(values: string | string[]) {
      const [periodPeriod] = values as string[];

      let value: AutomationLimit;
      try {
         value = this.state.value.updatePeriodPeriod(periodPeriod as ShortPeriod);
      } catch (e) {
         // Невалидное значение пропускаем
         return;
      }

      this.setState({ value });
   }

   @autobind
   private _onPeriodValueChange(rawValue: string) {
      const periodValue = parseInt(rawValue, 10);

      let value: AutomationLimit;
      try {
         value = this.state.value.updatePeriodValue(periodValue);
      } catch (e) {
         // Невалидное значение пропускаем
         return;
      }

      this.setState({ value });
   }

   @autobind
   private _onSubmit(e: SyntheticEvent) {
      e.preventDefault(); // prevent page reloading on form submit

      this.props.onSubmit(e, this.state.value);
   }

   @autobind
   private _onTimesChange(rawValue: string) {
      const times = parseInt(rawValue, 10);
      if (isNaN(times)) {
         return;
      }

      let value: AutomationLimit;
      try {
         value = this.state.value.updateTimes(times);
      } catch (e) {
         // Невалидное значение пропускаем
         return;
      }

      this.setState({ value });
   }
}
