import {
   clearParams,
   DISMISS_REASON,
   FormFieldType,
   FormLayout,
   IFormField,
   IModalProps,
   ModalLayout,
   useDismounted,
   toasts,
} from '@yandex-infracloud-ui/libs';
import React, { SyntheticEvent, useCallback, useState } from 'react';
import { forkJoin } from 'rxjs';
import { takeUntil } from 'rxjs/operators';

import {
   checkField,
   deployConfigField,
   deployConfigPolicyField,
   deployNetworkField,
   deployTagsField,
   disableAdminRequestsField,
   ignoreCmsField,
   reasonField,
   withAutoHealingField,
} from '../../../../actions/host_actions/actions/commonFields';
import {
   DefaultDeployConfigPolicy,
   DefaultDeployNetwork,
   DefaultOperationState,
   DefaultProvisioner,
   IAddHostsParams,
   IHost,
   IMaintenanceProperties,
} from '../../../../models';
import { RestrictionEditorField, EnumField } from '../../../../rich_shared/form_fields';
import {
   IMaintenancePropertiesFieldSubConfig,
   MaintenancePropertiesField,
   maintenancePropertiesFields,
} from '../../../../rich_shared/form_fields/MaintenancePropertiesField';
import { hostApi } from '../../../../services';
import { DevJson, ProjectName } from '../../../../shared';

type Host = Pick<IHost, 'inv' | 'name' | 'state' | 'status' | 'status_author'>[];

interface Props extends IModalProps<Host> {
   id: string;
}

const stateIsFree = (p: IAddHostsParams) => p.state === 'free';
const isOptionHidden = (p: IAddHostsParams) => stateIsFree(p) || p.instant;

const fields: IFormField<IAddHostsParams>[] = [
   {
      label: 'Inventory numbers or FQDNs',
      name: '_hosts',
      placeholder: 'Separated by comma, space or line by line',
      type: FormFieldType.StringSet,
   },
   {
      label: 'State',
      name: 'state',
      options: [
         { value: 'assigned', title: 'Assigned' },
         { value: 'free', title: 'Free' },
         { value: 'maintenance', title: 'Maintenance' },
      ],
      type: FormFieldType.Custom,
      component: EnumField,
   },
   {
      component: MaintenancePropertiesField,
      isHidden: p => p.state !== 'maintenance',
      label: 'Maintenance properties',
      name: 'maintenance_properties',
      subConfig: {
         skipFields: ['cms_task_action'],
      } as IMaintenancePropertiesFieldSubConfig,
      type: FormFieldType.Custom,
   },
   {
      ...deployConfigField,
      isHidden: stateIsFree,
   },
   {
      ...deployConfigPolicyField,
      isHidden: stateIsFree,
   },
   {
      ...deployNetworkField,
      isHidden: p => stateIsFree(p) || deployNetworkField.isHidden!(p),
   },
   {
      ...deployTagsField,
      isHidden: p => stateIsFree(p) || deployTagsField.isHidden!(p),
   },
   {
      isHidden: stateIsFree,
      label: 'Restrictions',
      name: '_restrictionMode',
      options: [
         { value: 'default', title: 'Project restrictions' },
         { value: 'manual', title: 'Manual' },
      ],
      type: FormFieldType.Custom,
      component: EnumField,
   },
   {
      component: RestrictionEditorField,
      isHidden: p => stateIsFree(p) || p._restrictionMode === 'default',
      name: 'restrictions',
      type: FormFieldType.Custom,
   },
   {
      isHidden: p => p.state !== 'assigned',
      label: 'Status',
      name: 'status',
      options: [
         { value: 'ready', title: 'Ready' },
         { value: 'manual', title: 'Manual' },
         { value: 'dead', title: 'Dead' },
      ],
      type: FormFieldType.Custom,
      component: EnumField,
   },

   reasonField,

   // Options
   {
      checkboxLabel: 'Do not create the task',
      isHidden: stateIsFree,
      name: 'instant',
      type: FormFieldType.Boolean,
   },
   {
      checkboxLabel: 'Setup DNS',
      isHidden: isOptionHidden,
      name: 'dns',
      type: FormFieldType.Boolean,
   },
   {
      ...checkField,
      isHidden: p => isOptionHidden(p) || checkField.isHidden!(p),
   },
   {
      ...withAutoHealingField,
      isHidden: p => isOptionHidden(p) || withAutoHealingField.isHidden!(p),
   },
   {
      ...disableAdminRequestsField,
      isHidden: p => isOptionHidden(p) || disableAdminRequestsField.isHidden!(p),
   },
   {
      ...ignoreCmsField,
      isHidden: p => isOptionHidden(p) || ignoreCmsField.isHidden!(p),
   },

   // hidden
   { name: 'config', type: FormFieldType.Hidden, isHidden: stateIsFree },
   { name: 'project', type: FormFieldType.Hidden },
   { name: 'provisioner', type: FormFieldType.Hidden, isHidden: p => stateIsFree(p) || !p.config },
];

const clearAddHostParams = (p: IAddHostsParams): Partial<IAddHostsParams> => {
   const cleaned = clearParams(fields, p);
   if (p._restrictionMode === 'manual') {
      cleaned.restrictions = p.restrictions;
   }

   if (cleaned.maintenance_properties) {
      cleaned.maintenance_properties = clearParams(
         maintenancePropertiesFields,
         cleaned.maintenance_properties,
      ) as IMaintenanceProperties;
   }

   return cleaned;
};

export const AddHostsToProjectModal: React.FC<Props> = React.memo(({ id, ok, cancel }) => {
   // hooks
   const dismounted = useDismounted();

   const [params, setParams] = useState<IAddHostsParams>({
      _hosts: new Set(),
      _restrictionMode: 'default',
      check: true,
      config: '',
      deploy_config_policy: DefaultDeployConfigPolicy,
      deploy_network: DefaultDeployNetwork,
      deploy_tags: new Set(),
      disable_admin_requests: false,
      dns: false,
      ignore_cms: false,
      ignore_maintenance: false,
      instant: false,
      maintenance_properties: {
         _timeoutTime: null,
         operation_state: DefaultOperationState,
         power_off: false,
         ticket_key: '',
         timeout_status: 'ready',
         timeout_time: null,
      },
      project: id,
      provisioner: DefaultProvisioner,
      reason: '',
      restrictions: new Set(),
      state: 'assigned',
      status: 'ready',
      with_auto_healing: true,
   });
   const [cleaned, setCleaned] = useState(clearAddHostParams(params));

   // handlers
   const handleSubmit = () => {
      forkJoin(hostApi.bulkAdd(params._hosts, cleaned))
         .pipe(takeUntil(dismounted))
         .subscribe(r => ok(r), toasts.handleApiError('Adding hosts'));
   };

   const updateParams = (e: SyntheticEvent | null, p: IAddHostsParams) => {
      p.config = p._deployConfig ? p._deployConfig.config : '';
      p.provisioner = p._deployConfig ? p._deployConfig.provisioner : DefaultProvisioner;

      setParams(p);
      setCleaned(clearAddHostParams(p));
   };

   const dismiss = useCallback(() => cancel(DISMISS_REASON), [cancel]);

   // render
   const title = (
      <>
         Add hosts to <ProjectName id={id} />
      </>
   );

   return (
      <ModalLayout title={title} okLabel={'Confirm'} onOk={handleSubmit} onDismiss={dismiss}>
         <FormLayout fields={fields} value={params} onChange={updateParams} />
         <DevJson>{cleaned}</DevJson>
      </ModalLayout>
   );
});

AddHostsToProjectModal.displayName = 'AddHostsToProjectModal';
