import {
   DISMISS_REASON,
   formatNumber,
   globalHotKeys,
   Keys,
   modalService,
   plural,
   toasts,
   useDismounted,
} from '@yandex-infracloud-ui/libs';
import * as React from 'react';
import { useCallback, useEffect } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import { Route, RouteComponentProps, Switch } from 'react-router';
import { takeUntil } from 'rxjs/operators';

import { AddedHost, ROUTE_LINKS } from '../../models';
import { ProjectSelect2 } from '../../rich_shared/ProjectSelect2/components/ProjectSelect2';
import { ProjectsHeader } from '../../rich_shared/ProjectSelect2/components/ProjectsHeader';
import { ITab, LegoButton, Page, Sidebar, SidebarDirection, SidebarToggleButton, Tabs } from '../../shared';
import { singleSelectedMyProjectIdSelector } from '../../state';
import { idSetToSet } from '../../state/commonModels';
import { addProject, projectsSlice } from '../../state/projects';
import { RootState } from '../../state/store';

import { DefaultNewProjectMode, NewProjectModalWrapper } from '../project/NewProjectModalWrapper';
import { AddHostsToProjectModal } from './hosts/components/AddHostsToProjectModal';

import styles from './index.module.css';
import { PROJECT_TABS, ROUTES } from './routes';

const tabs = PROJECT_TABS.map(
   r =>
      ({
         exact: false,
         header: r.name ? r.name.toUpperCase() : r.name,
         id: r.path,
         url: r.path,
      } as ITab),
);

export const ProjectsIndexScreen = React.memo(({ location, history, match }: RouteComponentProps) => {
   //region hooks
   const dismounted = useDismounted();
   const { selectedProjects, sidebarCollapsed } = useSelector((s: RootState) => ({
      selectedProjects: idSetToSet(s.projects.selectedIds),
      sidebarCollapsed: s.projects.isCollapsed,
   }));
   const dispatch = useDispatch();
   const singleSelectedProjectId = useSelector(singleSelectedMyProjectIdSelector);
   //endregion

   useEffect(
      () =>
         globalHotKeys.register({
            action() {
               dispatch(projectsSlice.actions.toggleSidebar());
            },
            help: 'Toggle sidebar with project list',
            hotKey: {
               code: Keys.KeyP,
            },
         }),
      [dispatch],
   );

   //region handlers
   const createOrCloneProject = useCallback(() => {
      modalService
         .open(NewProjectModalWrapper, { mode: DefaultNewProjectMode }, false)
         .pipe(takeUntil(dismounted))
         .subscribe(
            project => {
               dispatch(
                  addProject({
                     ...project,
                     deploy_tags: Array.from(project.deploy_tags ?? []),
                     tags: Array.from(project.tags ?? []),
                  }),
               );

               history.push(ROUTE_LINKS.project(project.id));
            },
            reason => {
               if (reason !== DISMISS_REASON) {
                  toasts.apiError('Project creation', reason);
               }
            },
         );
   }, [dismounted, dispatch, history]);

   const handleAddHosts = useCallback(() => {
      modalService
         .open(AddHostsToProjectModal, { id: singleSelectedProjectId! })
         .pipe(takeUntil(dismounted))
         .subscribe(
            hosts => {
               const invs = hosts
                  .filter((r: any) => r.success)
                  .map((r: any) => r.response as AddedHost)
                  .map((h: any) => h.inv.toString());

               if (invs.length > 0) {
                  toasts.success(
                     `${formatNumber(invs.length)} ${plural(invs.length, 'host has', 'hosts have')} been added`,
                  );
                  history.push(ROUTE_LINKS.hostList({ fqdn: invs.join(' ') }));
               }
            },
            () => null,
         );
   }, [dismounted, history, singleSelectedProjectId]);
   //endregion

   //region render

   const sidebarContent = <ProjectSelect2 />;

   const sidebarFooter = (
      <>
         <SidebarToggleButton direction={SidebarDirection.Collapse} />

         <div>
            {singleSelectedProjectId === null ? null : (
               <LegoButton cls={styles.footerButton} theme={'action'} onClick={handleAddHosts}>
                  Add hosts
               </LegoButton>
            )}

            <LegoButton
               theme={'action'}
               cls={styles.footerButton}
               onClick={createOrCloneProject}
               controlAttrs={{ 'data-e2e': 'NewProjectButton' }}
            >
               Create project
            </LegoButton>
         </div>
      </>
   );

   const header = (
      <>
         <ProjectsHeader className={styles.header} history={history} location={location} match={match} />

         <Tabs items={tabs} commonParams={{ project: selectedProjects }} />
      </>
   );

   const routes = ROUTES.map(r => <Route key={r.path as string} {...r} />);

   return (
      <Sidebar isCollapsed={sidebarCollapsed} footer={sidebarFooter} content={sidebarContent}>
         <Page header={header}>
            <Switch>{routes}</Switch>
         </Page>
      </Sidebar>
   );
   //endregion
});

ProjectsIndexScreen.displayName = 'ProjectsIndexScreen';
