import { fromQuery, isEqual, parseDate } from '@yandex-infracloud-ui/libs-next';
import { ParsedQuery } from 'query-string';
import * as React from 'react';
import { ReactNode } from 'react';
import { Redirect, RouteComponentProps } from 'react-router';

import { ROUTE_LINKS } from '../models';
import { getDefaultLogFilters, logFiltersToUrlParams } from '../rich_shared';
import { getDefaultHostFilters, hostFiltersToUrlParams } from '../rich_shared/host_list';

import { sentry } from '../services';

type RedirectHandler = (id: string, query: ParsedQuery) => string | null;

const OLD_ROUTES: [RegExp, RedirectHandler][] = [
   // Automations plots
   [/^\/automation-plots$/, () => ROUTE_LINKS.automationPlots()],
   [/^\/automation-plots\/(.+?)$/, id => ROUTE_LINKS.automationPlot(id)],
   // Страница настроек проекта
   [/^\/project\/(.+?)\/info\/$/, id => ROUTE_LINKS.project(id)],
   // Аудитный лог
   [/^\/log\/$/, handleAuditLog],
   [/^\/log\/(.+?)\/$/, handleAuditLog],
   // Список хостов
   [/^\/project\/(.+?)\/$/, handleHostList],
   [/^\/$/, handleHostList],
];

export function redirects({ location }: RouteComponentProps<any>): ReactNode {
   // new balancer TODO remove this in may 2019
   if (window.location.hostname === 'wall-e-dev.n.yandex-team.ru') {
      window.location.hostname = 'wall-e-dev.in.yandex-team.ru';

      return null;
   }

   const path = location.pathname;
   const query = fromQuery(location.search, { arrayFormat: 'index' });

   // console.log({path, query});

   for (const [re, newRoute] of OLD_ROUTES) {
      const matches = path.match(re);
      if (matches) {
         const to = newRoute(matches[1], query);
         if (!to) {
            return null;
         }

         sentry.sendRedirect(window.document.referrer, path, to);

         return <Redirect to={to} />;
      }
   }

   return null;
}

function handleHostList(project: string | undefined, query: ParsedQuery): string | null {
   // Отдельный хост по инвентарному номеру: "прямая" ссылка на хост с других систем ()
   if (typeof query.hosts === 'string' && /^\d+$/.test(query.hosts)) {
      return ROUTE_LINKS.host(query.hosts);
   }

   const filters = getDefaultHostFilters();

   // project
   if (project) {
      filters.project.add(project);
   }

   // hosts
   if (query.hosts) {
      filters.fqdn = query.hosts as string;
   }

   // switch
   if (query.switch) {
      filters.switch.add(query.switch as string);
   }

   // deploy config
   if (query.config) {
      filters.config.add(query.config as string);
   }

   // status
   if (Array.isArray(query.status)) {
      query.status.forEach(s => filters.status.add(s));
   }

   // health
   if (Array.isArray(query.health)) {
      query.health.forEach(h => filters.health.add(h));
   }

   // state
   if (Array.isArray(query.state)) {
      query.state.forEach(s => filters.state.add(s));
   }

   // restrictions
   if (Array.isArray(query.restrictions)) {
      query.restrictions.forEach(r => filters.restrictions.add(r));
   }

   // location
   if (query.physical_location) {
      filters.locations.add(query.physical_location as string);
   }

   // console.log({query, filters});

   if (isEqual(filters, getDefaultHostFilters())) {
      return null;
   }

   return ROUTE_LINKS.hostList(hostFiltersToUrlParams(filters));
}

function handleAuditLog(project: string | undefined, query: ParsedQuery): string | null {
   // Аудитный лог отдельного хоста
   if (typeof query.host === 'string') {
      return ROUTE_LINKS.hostAuditLog(query.host);
   }

   const filters = getDefaultLogFilters();

   // project
   if (project) {
      filters.project.add(project);
   }

   // status
   if (Array.isArray(query.status)) {
      query.status.forEach(s => filters.status.add(s));
   }

   // start_time
   if (query.start_time) {
      filters.from = parseDate(query.start_time as string);
   }

   // end_time
   if (query.end_time) {
      filters.to = parseDate(query.end_time as string);
   }

   // issuer
   if (query.issuer) {
      filters.issuer.add(query.issuer as string);
   }

   // type
   if (Array.isArray(query.type)) {
      query.type.forEach(t => filters.type.add(t));
   }

   // console.log({query, filters});

   if (isEqual(filters, getDefaultLogFilters())) {
      return null;
   }

   return ROUTE_LINKS.auditLog(logFiltersToUrlParams(filters));
}
