import { parseHostQuery } from './api_helpers';

describe('api_helpers', () => {
   describe('parseHostQuery', () => {
      it('should parse empty', () => {
         expect(parseHostQuery('')).toEqual({
            invs: [],
            names: [],
            patterns: [],
            tags: [],
            uuids: [],
         });
      });

      it('should parse with +', () => {
         const expected = {
            invs: [],
            names: ['old-bering.search.yandex.net', 'weasel.search.yandex.net'],
            patterns: [],
            tags: [],
            uuids: [],
         };

         expect(parseHostQuery('old-bering.search.yandex.net+weasel.search.yandex.net')).toEqual(expected);

         expect(parseHostQuery('+old-bering.search.yandex.net +weasel.search.yandex.net')).toEqual(expected);

         expect(parseHostQuery('old-bering.search.yandex.net +weasel.search.yandex.net')).toEqual(expected);

         expect(parseHostQuery('old-bering.search.yandex.net+ weasel.search.yandex.net')).toEqual(expected);

         expect(parseHostQuery('old-bering.search.yandex.net + weasel.search.yandex.net')).toEqual(expected);
      });

      it('should parse without tags', () => {
         expect(parseHostQuery('vla')).toEqual({
            invs: [],
            names: ['vla'],
            patterns: [],
            tags: [],
            uuids: [],
         });
      });

      it('should parse with empty tag', () => {
         expect(parseHostQuery('a #')).toEqual({
            invs: [],
            names: ['a'],
            patterns: [],
            tags: [],
            uuids: [],
         });
      });

      it('should parse with one tag', () => {
         expect(parseHostQuery('#mtn')).toEqual({
            invs: [],
            names: [],
            patterns: [],
            tags: ['mtn'],
            uuids: [],
         });
      });

      it('should parse with two tags', () => {
         expect(parseHostQuery('vla1 #mtn #rtc')).toEqual({
            invs: [],
            names: ['vla1'],
            patterns: [],
            tags: ['mtn', 'rtc'],
            uuids: [],
         });
      });

      it('should parse inventory number', () => {
         expect(parseHostQuery('100164782')).toEqual({
            invs: [100164782],
            names: [],
            patterns: [],
            tags: [],
            uuids: [],
         });
      });

      it('should parse all together', () => {
         expect(parseHostQuery('vla1-1123 100164782 #mtn f2cb79f69c634c78be5c878b7c9e2580')).toEqual({
            invs: [100164782],
            names: ['vla1-1123'],
            patterns: [],
            tags: ['mtn'],
            uuids: ['f2cb79f69c634c78be5c878b7c9e2580'],
         });
      });

      it('should parse two fqdn', () => {
         expect(parseHostQuery('sandbox530.search.yandex.net iva5-y98-01.cloud.yandex.net')).toEqual({
            invs: [],
            names: ['sandbox530.search.yandex.net', 'iva5-y98-01.cloud.yandex.net'],
            patterns: [],
            tags: [],
            uuids: [],
         });
      });

      it('should parse patterns', () => {
         expect(parseHostQuery('sandbox*.search.yandex.net iva5-y98-01.cloud.yandex.net')).toEqual({
            invs: [],
            names: ['iva5-y98-01.cloud.yandex.net'],
            patterns: ['sandbox*.search.yandex.net'],
            tags: [],
            uuids: [],
         });
      });

      it('should parse uuids', () => {
         expect(parseHostQuery('aa07858f00e14f06a163cc3dba595f87')).toEqual({
            invs: [],
            names: [],
            patterns: [],
            tags: [],
            uuids: ['aa07858f00e14f06a163cc3dba595f87'],
         });
      });

      it('should parse two uuids', () => {
         expect(parseHostQuery('aa07858f00e14f06a163cc3dba595f87 123e4567-e89b-12d3-a456-426655440000')).toEqual({
            invs: [],
            names: [],
            patterns: [],
            tags: [],
            uuids: ['aa07858f00e14f06a163cc3dba595f87', '123e4567-e89b-12d3-a456-426655440000'],
         });
      });
   });
});
