import { arrayToSet, fromTimestamp, HttpMethod, PER_PAGE_LOG } from '@yandex-infracloud-ui/libs-next';
import { Observable } from 'rxjs';
import { tap } from 'rxjs/operators';

import { allLogItemFields, IListResult, ILogItem, ILogRequest, shortLogItemFields, SYSTEM_USERS } from '../../models';
import { ILogFilters } from '../../rich_shared';
import { config } from '../config';

import { parseHostQuery } from './api_helpers';
import { WalleBaseApi } from './base_api';

export class AuditLogApi extends WalleBaseApi {
   public getList(filters: ILogFilters, cursor?: Date): Observable<IListResult<ILogItem>> {
      const { invs, names, uuids } = parseHostQuery(filters.fqdn);

      const issuer = Array.from(filters.issuer).map(login =>
         login.endsWith('@') || SYSTEM_USERS.has(login) ? login : `${login}@`,
      );

      const params: ILogRequest = _fixTimeFilters({
         cursor,
         end_time: filters.to || undefined, // null нельзя, т.к. тогда будет пустой ключ в запросе
         fields: shortLogItemFields,
         host_inv: new Set(invs),
         host_name: new Set(names),
         host_uuid: new Set(uuids),
         issuer: arrayToSet(issuer),
         limit: PER_PAGE_LOG,
         preorder: filters.preorder,
         project: filters.project,
         scenario_id: filters.scenario_id,
         start_time: filters.from || undefined, // null нельзя, т.к. тогда будет пустой ключ в запросе
         status: filters.status,
         type: filters.type,
      });

      return this.request<ILogRequest, void, IListResult<ILogItem>>(HttpMethod.GET, '', params).pipe(
         tap(resp => resp.result.forEach(_fillLogItemFields)),
      );
   }

   public getById(id: string): Observable<ILogItem> {
      const params = {
         fields: allLogItemFields,
      };

      return this.request<typeof params, void, ILogItem>(HttpMethod.GET, id, params).pipe(tap(_fillLogItemFields));
   }
}

function _fillLogItemFields(item: ILogItem): ILogItem {
   item.time = fromTimestamp(item.time);
   item.status_time = fromTimestamp(item.status_time);

   return item;
}

function _fixTimeFilters(params: ILogRequest): ILogRequest {
   const newParams = { ...params };

   if (newParams.start_time) {
      newParams.start_time = new Date(Math.floor(newParams.start_time.getTime() / 1000) * 1000);
   }

   if (newParams.end_time) {
      newParams.end_time = new Date(Math.ceil(newParams.end_time.getTime() / 1000) * 1000);
   }

   return newParams;
}

export const auditLogApi = new AuditLogApi(`${config.walleApi}/v1/audit-log`);
