import { HttpMethod } from '@yandex-infracloud-ui/libs';
import { Observable, of } from 'rxjs';
import { map, tap } from 'rxjs/operators';
import {
   IBotProject,
   IBoxes,
   IConfig,
   IConstants,
   IDeployConfigs,
   IGlobalSettings,
   IHbfProject,
   IListResult,
   ILocation,
   IUserInfo,
} from '../../models';
import { config } from '../config';

import { ApiCache } from './api_cache';
import { WalleBaseApi } from './base_api';

class DictApi extends WalleBaseApi {
   /**
    * То же, что и _botProjectMap, но наоборот
    */
   private _abcServiceMap?: Map<number, number>;

   /**
    * Карта соответствий ABC-сервиса Bot проекту.
    *
    * Ключ - ID ABC сервиса, значение - ID Bot project
    */
   private _botProjectMap?: Map<number, number>;

   private _cache = new ApiCache();

   public getBoxes(): Observable<IBoxes> {
      return this._cache.request('boxes', () => this.request(HttpMethod.GET, 'boxes'));
   }

   public getConstants(): Observable<IConstants> {
      return this._cache.request('constants', () => this.request(HttpMethod.GET, 'constants'));
   }

   public getConfig(): Observable<IConfig> {
      return this._cache.request('config', () => this.request(HttpMethod.GET, 'config'));
   }

   public getCurrentUser(skipCache = false): Observable<IUserInfo> {
      return this._cache.request('user', () => this.request(HttpMethod.GET, 'user'), skipCache);
   }

   public getCsrfToken(): Observable<string> {
      return this.request<void, void, { csrf_token: string }>(HttpMethod.GET, 'csrf-token').pipe(
         map(resp => resp.csrf_token),
      );
   }

   public getLocations(project?: string): Observable<IListResult<ILocation>> {
      return this._cache.request(`locations_${project ?? 'all'}`, () =>
         this.request(HttpMethod.GET, 'physical-location-tree', { project }),
      );
   }

   public getDeployConfigs(): Observable<IDeployConfigs> {
      return this._cache.request('deployConfigs', () => this.request(HttpMethod.GET, 'deploy-configs'));
   }

   public getSettings(): Observable<IGlobalSettings> {
      return this._cache.request('settings', () => this.request(HttpMethod.GET, 'settings'));
   }

   public getHbfProjects(): Observable<IHbfProject[]> {
      return this._cache.request('hbfProjects', () =>
         this.request<any, any, IListResult<IHbfProject>>(HttpMethod.GET, 'hbf-projects').pipe(
            map(resp => resp.result),
         ),
      );
   }

   public getHbfProject(id: string): Observable<IHbfProject | null> {
      const normalizedId = id.replace(/^0x/, '');

      return this.getHbfProjects().pipe(map(resp => resp.find(p => p.id === normalizedId) || null));
   }

   /**
    * Возвращаемое значение в качестве ключа содержит ID ABC сервиса, а значения - ID Bot project
    */
   public getBotProjectMap(): Observable<Map<number, number>> {
      if (this._botProjectMap) {
         return of(this._botProjectMap);
      }

      return this._fillBotAndABCMaps().pipe(map(() => this._botProjectMap!));
   }

   /**
    * Возвращаемое значение в качестве ключа содержит ID Bot project, а значения - ID ABC сервиса
    */
   public getAbcServiceProjectMap(): Observable<Map<number, number>> {
      if (this._abcServiceMap) {
         return of(this._abcServiceMap);
      }

      return this._fillBotAndABCMaps().pipe(map(() => this._abcServiceMap!));
   }

   /**
    * Мутирует idMap, заполняя его значениями рекурсивно (обходя дерево projects)
    */
   private _fillAbcToBotMap(idMap: Map<number, number>, projects: IBotProject[]): void {
      for (const project of projects) {
         idMap.set(project.planner_id, project.project_id);
      }
   }

   private _fillBotAndABCMaps() {
      return this.getBotProjects().pipe(
         tap(resp => {
            this._botProjectMap = new Map();
            this._fillAbcToBotMap(this._botProjectMap, resp);

            this._abcServiceMap = new Map();
            this._botProjectMap.forEach((v, k) => this._abcServiceMap!.set(v, k));
         }),
      );
   }

   public getBotProjects(): Observable<IBotProject[]> {
      return this._cache.request('bot-projects', () =>
         this.request<any, any, IListResult<IBotProject>>(HttpMethod.GET, 'bot-projects').pipe(
            map(obj => obj.result),
            map(res => this._getFlatBotProjects(res)),
            map(res =>
               res.map(e => ({
                  ...e,
                  isOld: e.en_description.startsWith('OLD_'),
                  searchIndex: e.en_description.toLowerCase() + String(e.planner_id) + String(e.project_id),
               })),
            ),
         ),
      );
   }

   private _getFlatBotProjects(items: IBotProject[]): IBotProject[] {
      // @ts-ignore
      const projects: IBotProject[] = [...items];
      const result: IBotProject[] = [];

      while (projects.length > 0) {
         const project = projects.shift();

         if (project?.subprojects && project.subprojects.length > 0) projects.unshift(...project.subprojects);

         result.push(project!);
      }

      return result;
   }
}

export const dictApi = new DictApi(`${config.walleApi}/v1`);
