import { arrayToSet, fillSetFields, fromTimestamp, HttpMethod } from '@yandex-infracloud-ui/libs-next';
import { BehaviorSubject, forkJoin, Observable, of } from 'rxjs';
import { last, map, mergeMap, scan, tap } from 'rxjs/operators';

import {
   AddedHost,
   allHostFields,
   HostLogType,
   HostPowerStatusResponse,
   IAddHostsParams,
   IConfigureDeployHostParams,
   IEditExtraVlansHostParams,
   IHost,
   IHostActionParams,
   IHostFilters,
   IHostListPostRequest,
   IHostListRequest,
   IListResult,
   IMetaStatusValues,
   IRemoveHostParams,
   IUpdateHostRestrictionParams,
   IVlanConfig,
   MetaStatusButtons,
   OwnershipValue,
   shortHostFields,
} from '../../models';
import { config } from '../config';

import { parseHostQuery, splitParams } from './api_helpers';
import { WalleBaseApi } from './base_api';

interface GetProjectListOptions {
   fields?: string[];
   filters: IHostFilters;
   limit: number;
   include_shadow?: boolean;
   nextCursor: number | undefined;
   useOwnershipFilter: boolean;
   userProjects?: string[];
   sortBy?: string;
}

class HostApi extends WalleBaseApi {
   public getList({
      fields = shortHostFields,
      filters,
      limit,
      include_shadow = true,
      nextCursor,
      sortBy = '',
      useOwnershipFilter,
      userProjects,
   }: GetProjectListOptions): Observable<IListResult<IHost>> {
      // Не делать запрос, если запрошен пустой список инвентарников
      if (filters.ids && filters.ids.size === 0) {
         return of({ result: [], total: 0 });
      }

      const { invs, names, patterns, tags, uuids } = parseHostQuery(filters.fqdn);

      // Если в запросе не указаны конкретные инвентарники, то используются filters.ids
      if (filters.ids && invs.length === 0) {
         invs.push(...Array.from(filters.ids));
      }

      let params: IHostListRequest = {
         config: filters.config,
         fields,
         health: filters.health,
         include_shadow,
         limit,
         physical_location: filters.locations,
         project: this._getProjectFilter(filters, useOwnershipFilter, userProjects),
         restrictions: filters.restrictions,
         state: filters.state,
         status: filters.status,
         switch: filters.switch,
         tags: new Set(tags),
         type: filters.type,
      };

      params = sortBy.length > 0 ? { ...params, 'sort-by': sortBy } : { ...params, cursor: nextCursor };

      if (invs.length > 0 || names.length > 1 || patterns.length > 0 || uuids.length > 0) {
         const body: IHostListPostRequest = {
            invs,
            names,
            patterns,
            uuids,
         };

         return this.request(HttpMethod.POST, 'get-hosts', params, body);
      } else {
         params.name = names.join('');

         return this.request(HttpMethod.GET, 'hosts', params);
      }
   }

   public getStat(
      filters: IHostFilters,
      useOwnershipFilter: boolean,
      userProjects: string[] | undefined,
   ): Observable<IMetaStatusValues> {
      const metaStatuses = MetaStatusButtons.config.map(b => b.id);

      const requests = metaStatuses.map(status => {
         const patchedFilters = MetaStatusButtons.patchFilters(filters, status);

         return this.getList({
            fields: [],
            filters: patchedFilters,
            limit: 0,
            nextCursor: undefined,
            useOwnershipFilter,
            userProjects,
         });
      });

      const emptyValues: IMetaStatusValues = { all: 0, dead: 0, error: 0, processing: 0, queue: 0 };

      return forkJoin(...requests).pipe(
         map((responses: IListResult<IHost>[]) => {
            return responses.reduce((acc, item, index) => {
               const metaStatus = metaStatuses[index];

               acc[metaStatus] = item.total!;

               return acc;
            }, emptyValues);
         }),
      );
   }

   public getByFQDN(fqdn: string): Observable<IHost> {
      const params: Partial<IHostListRequest> = {
         fields: allHostFields,
      };

      return this.request<Partial<IHostListRequest>, void, IHost>(HttpMethod.GET, `hosts/${fqdn}`, params).pipe(
         tap(_fillHostFields),
      );
   }

   public getVLANs(uuid: string): Observable<IVlanConfig> {
      return this.request(HttpMethod.GET, `hosts/${uuid}/current-configuration`);
   }

   public getPowerStatus(uuid: string): Observable<boolean> {
      return this.request<void, void, HostPowerStatusResponse>(HttpMethod.GET, `hosts/${uuid}/power-status`).pipe(
         map(resp => resp.is_powered_on),
      );
   }

   public getLogLink(uuid: string, type: HostLogType): string {
      return `${this._apiPrefix}/hosts/${uuid}/${type}-log`;
   }

   public getLogTail(uuid: string, type: HostLogType): Observable<string> {
      const params = { tail_bytes: 5000 };

      return this.request(HttpMethod.GET, `hosts/${uuid}/${type}-log`, params, undefined, false);
   }

   public getCoronerLogTail(host: string): Observable<string> {
      const params = { tail_bytes: 5000, host };

      return this.request(HttpMethod.GET, `/external_api/coroner/v1/raw`, params, undefined, false);
   }

   public getBotConfiguration(inv: number): Observable<string> {
      return this.request(HttpMethod.GET, `/external_api/bot/consistof.php`, { inv }, undefined, false);
   }

   /**+
    *
    * @param uuid
    * @param action
    * @param params
    * @param getFields Список полей, которые будет отправлены не в теле, а в URL запроса
    */
   public doSimpleAction<T extends Partial<IHostActionParams>, R>(
      uuid: string,
      action: string,
      params: T,
      getFields: (keyof T)[] = ['ignore_maintenance'],
   ): Observable<R> {
      const [getParams, bodyParams] = splitParams(params, getFields);

      return this.request(HttpMethod.POST, `hosts/${uuid}/${action}`, getParams, bodyParams);
   }

   public actionWithMultiHosts<T>(action: string, body: T): Observable<{ link: string }> {
      return this.request(HttpMethod.POST, `${action}`, {}, body);
   }

   public changeDeployConfigAction(uuid: string, params: Partial<IConfigureDeployHostParams>): Observable<IHost> {
      const [getParams, bodyParams] = splitParams(params, ['ignore_maintenance']);
      const method = params.config ? HttpMethod.PUT : HttpMethod.DELETE;

      return this.request(method, `hosts/${uuid}/deploy_config`, getParams, bodyParams);
   }

   public changeExtraVlans(uuid: string, params: Partial<IEditExtraVlansHostParams>): Observable<IHost> {
      const [getParams, bodyParams] = splitParams(params, ['ignore_maintenance']);

      return this.request(HttpMethod.PUT, `hosts/${uuid}/extra_vlans`, getParams, bodyParams);
   }

   public remove(uuid: string, params: Partial<IRemoveHostParams>): Observable<IHost> {
      const [bodyParams, getParams] = splitParams(params, ['reason']);

      return this.request(HttpMethod.DELETE, `hosts/${uuid}`, getParams, bodyParams);
   }

   public updateRestrictions(uuid: string, params: IUpdateHostRestrictionParams): Observable<IHost> {
      const [getParams, bodyParams] = splitParams(params, ['ignore_maintenance']);

      return this.request(HttpMethod.POST, `hosts/${uuid}`, getParams, bodyParams);
   }

   /**
    * В отличие от прочих методов, этот возвращает массив Observable,
    * т.к. для каждого хоста делается отдельный запрос.
    */
   public bulkAdd(hosts: Set<string>, params: Partial<IAddHostsParams>): Observable<AddedHost>[] {
      return Array.from(hosts).map(host => {
         const body = { ...params };

         if (/^\d+$/.test(host)) {
            body.inv = parseInt(host, 10);
         } else {
            body.name = host;
         }

         return this.request(HttpMethod.POST, 'hosts', {}, body);
      });
   }

   public getAll(
      filters: IHostFilters,
      useOwnershipFilter: boolean,
      userProjects: string[] | undefined,
      total: number,
      fields = ['name', 'uuid', 'project', 'state'],
   ): Observable<IHost[]> {
      const limit = 10000; // Maximum allowed by API
      const loadChunk = (nextCursor: number) =>
         this.getList({
            fields,
            filters,
            limit,
            nextCursor,
            useOwnershipFilter,
            userProjects,
         });

      const iterator = new BehaviorSubject(0);

      return iterator.pipe(
         mergeMap(cursor =>
            loadChunk(cursor).pipe(
               tap(resp => {
                  if (resp.next_cursor) {
                     iterator.next(resp.next_cursor);
                  } else {
                     iterator.complete();
                  }
               }),
            ),
         ),
         scan((all, current) => all.concat(current.result), [] as IHost[]),
         last(),
      );
   }

   private _getProjectFilter(
      filters: IHostFilters,
      useOwnershipFilter: boolean,
      userProjects: string[] | undefined,
   ): Set<string> {
      if (filters.project.size > 0) {
         return filters.project;
      }

      const useMyProjects = useOwnershipFilter && filters.ownership === OwnershipValue.My && userProjects !== undefined;

      return arrayToSet(useMyProjects ? userProjects : []);
   }
}

export function _fillHostFields(host: IHost): void {
   if (host.active_mac_time) {
      host.active_mac_time = fromTimestamp(host.active_mac_time);
   }

   if (host.location) {
      host.location = {
         ...host.location,
         network_timestamp: fromTimestamp(host.location.network_timestamp),
         physical_timestamp: fromTimestamp(host.location.physical_timestamp),
      };
   }

   fillSetFields(host, ['deploy_tags', 'restrictions']);
}

export const hostApi = new HostApi(`${config.walleApi}/v1`);
