import { HttpMethod, isEqual, isEmpty } from '@yandex-infracloud-ui/libs-next';
import { forkJoin, Observable, of } from 'rxjs';

import { allMaintenancePlotFields, IMaintenancePlot, IListResult, IPlotActionParams } from '../../models';
import { config } from '../config';

import { WalleBaseApi } from './base_api';

function _getPatchForFields<T>(original: T, edited: T, fields: (keyof T)[]): Partial<T> {
   const result: Partial<T> = {};

   for (const field of fields) {
      if (!isEqual(original[field], edited[field])) {
         result[field] = edited[field];
      }
   }

   return result;
}

class MaintenancePlotsApi extends WalleBaseApi {
   private _allFields = allMaintenancePlotFields;

   public getList(): Observable<IListResult<IMaintenancePlot>> {
      const params = {
         fields: this._allFields,
      };

      return this.request<typeof params, void, IListResult<IMaintenancePlot>>(HttpMethod.GET, '', params).pipe();
   }

   public getById(id: string): Observable<IMaintenancePlot> {
      const params = {
         fields: this._allFields,
      };

      return this.request<typeof params, void, IMaintenancePlot>(HttpMethod.GET, id, params);
   }

   public create(plot: IMaintenancePlot): Observable<IMaintenancePlot> {
      return this.request(HttpMethod.POST, '', null, plot);
   }

   public update(original: IMaintenancePlot, edited: IMaintenancePlot, reason: string): Observable<any> {
      const requests: Observable<any>[] = [
         of(null), // Фейковый запрос, чтобы при отсутствии изменений вернуть хоть что-нибудь
      ];

      const metaInfoPatch = _getPatchForFields(original, edited, ['meta_info']);
      if (!isEmpty(metaInfoPatch)) {
         requests.push(
            this.request(HttpMethod.PUT, `${original.id}/meta_info`, null, {
               ...metaInfoPatch.meta_info,
               reason,
            }),
         );
      }

      const commonPatch = _getPatchForFields(original, edited, ['common_settings']);
      if (!isEmpty(commonPatch)) {
         requests.push(
            this.request(HttpMethod.PUT, `${original.id}/common_settings`, null, {
               ...commonPatch.common_settings,
               reason,
            }),
         );
      }

      const scenariosPatch = _getPatchForFields(original, edited, ['scenarios_settings']);
      if (!isEmpty(scenariosPatch)) {
         requests.push(
            this.request(HttpMethod.PUT, `${original.id}/scenarios_settings`, null, {
               ...scenariosPatch,
               reason,
            }),
         );
      }

      return forkJoin(requests);
   }

   public remove(id: string, params: IPlotActionParams): Observable<void> {
      return this.request(HttpMethod.DELETE, id, null, params);
   }
}

export const maintenancePlotsApi = new MaintenancePlotsApi(`${config.walleApi}/v1/maintenance-plots`);
