import { fromTimestamp, HttpMethod, PER_PAGE_LOG } from '@yandex-infracloud-ui/libs-next';
import { Observable } from 'rxjs';
import { tap } from 'rxjs/operators';

import { IListResult, IOperationLogItem, IOperationLogRequest, operationLogItemFields } from '../../models';
import { IOperationLogFilters } from '../../rich_shared/OperationLog';
import { config } from '../config';

import { parseHostQuery } from './api_helpers';
import { WalleBaseApi } from './base_api';

class OperationLogApi extends WalleBaseApi {
   public getList(filters: IOperationLogFilters, cursor?: Date): Observable<IListResult<IOperationLogItem>> {
      const { invs, names, uuids } = parseHostQuery(filters.fqdn);

      const params: IOperationLogRequest = _fixTimeFilters({
         cursor,
         id: filters.id,
         end_time: filters.to || undefined, // null нельзя, т.к. тогда будет пустой ключ в запросе
         fields: operationLogItemFields,
         host_inv: new Set(invs),
         host_name: new Set(names),
         host_uuid: new Set(uuids),
         limit: PER_PAGE_LOG,
         project: filters.project,
         start_time: filters.from || undefined, // null нельзя, т.к. тогда будет пустой ключ в запросе
         type: filters.type,
         audit_log_id: filters.audit_log_id,
      });

      return this.request<IOperationLogRequest, void, IListResult<IOperationLogItem>>(HttpMethod.GET, '', params).pipe(
         tap(resp => resp.result.forEach(_fillLogItemFields)),
      );
   }
}

function _fillLogItemFields(item: IOperationLogItem): IOperationLogItem {
   item.time = fromTimestamp(item.time);
   item.status_time = fromTimestamp(item.status_time);

   return item;
}

function _fixTimeFilters(params: IOperationLogRequest): IOperationLogRequest {
   const newParams = { ...params };

   if (newParams.start_time) {
      newParams.start_time = new Date(Math.floor(newParams.start_time.getTime() / 1000) * 1000);
   }

   if (newParams.end_time) {
      newParams.end_time = new Date(Math.ceil(newParams.end_time.getTime() / 1000) * 1000);
   }

   return newParams;
}

export const operationLogApi = new OperationLogApi(`${config.walleApi}/v1/operation-log`);
