import {
   deepClone,
   fillSetFields,
   getSetDifference,
   HttpMethod,
   isEmpty,
   isEqual,
   omitFields,
   toTimestamp,
   uniqueBy,
} from '@yandex-infracloud-ui/libs';
import { forkJoin, Observable, of } from 'rxjs';
import { map, switchMap, tap } from 'rxjs/operators';

import {
   allProjectFields,
   AutomationLimit,
   AutomationType,
   CmsValue,
   createProjectOwners,
   DefaultDeployNetwork,
   IAutomationCreditsForm,
   IAutomationLimits,
   ICloneProjectParams,
   IConstants,
   ICustomCms,
   IDefaultCms,
   IdmRequest,
   IFsmHandbrakeProjectParams,
   IListResult,
   INewProject,
   IProject,
   IProjectActionParams,
   IProjectListRequest,
   ProjectCmsSettings,
   Provisioner,
   TimeoutMode,
   VlanSchema,
} from '../../models';

import { clearReportsParams } from '../../rich_shared/project_report_settings';

import { config } from '../config';
import { WalleBaseApi } from './base_api';
import { dictApi } from './dict_api';

class ProjectApi extends WalleBaseApi {
   private _allFields = allProjectFields;

   public getList(): Observable<IListResult<IProject>> {
      const params: IProjectListRequest = {
         fields: [
            'deploy_tags', // Для подсказок
            'dns_automation',
            'dns_domain',
            'fsm_handbrake.timeout_time',
            'type',
            'healing_automation',
            'id',
            'name',
            // 'owners', пока не нужен
            'reboot_via_ssh',
            'tags', // Для подсказок и фильтра
         ],
      };

      return this.request<IProjectListRequest, void, IListResult<IProject>>(HttpMethod.GET, '', params).pipe(
         tap(resp => resp.result.forEach(_fillProjectFields)),
      );
   }

   public getById(projectId: string, fields = this._allFields): Observable<IProject> {
      const params: IProjectListRequest = { fields };

      return forkJoin([
         dictApi.getConstants(),
         this.request<IProjectListRequest, void, IProject>(HttpMethod.GET, projectId, params),
      ]).pipe(
         map(([constants, project]) => {
            _fillProjectFields(project);
            _splitProjectAutomationLimits(project, constants);

            return project;
         }),
      );
   }

   /* Create without modifying input data */
   public fastCreate(project: INewProject): Observable<IProject> {
      return this.request<void, INewProject, IProject>(HttpMethod.POST, '', undefined, project).pipe(
         tap(_fillProjectFields),
      );
   }

   public create(project: INewProject): Observable<IProject> {
      const separateSavingFields: (keyof INewProject)[] = [
         'all_available_project_checks',
         'automation_limits',
         'host_limits',
         'manually_disabled_checks',
         'reports',

         // vlans
         'extra_vlans',
         'hbf_project_id',
         'ip_method',
         'native_vlan',
         'owned_vlans',
         'vlan_scheme',

         // owners
         '_owners',
      ];

      const directPost: INewProject = {
         ...omitFields(project, ...separateSavingFields, 'dns_automation', 'healing_automation', 'cms'),
         cms_settings: project?.cms,
         enable_dns_automation: project.dns_automation?.enabled,
         enable_healing_automation: project.healing_automation?.enabled,
         owners: config.isExternal ? undefined : project._owners?.toAdd,
         reason: project.reason, // just for readability
         type: project.type,
      };

      return this.request<void, INewProject, IProject>(HttpMethod.POST, '', undefined, directPost).pipe(
         switchMap(createdProject => this.getById(createdProject.id)),
         switchMap(createdProject => {
            const original = deepClone(createdProject);

            for (const field of separateSavingFields) {
               switch (field) {
                  case 'automation_limits':
                     for (const [limit, value] of Object.entries(project.automation_limits!)) {
                        createdProject.automation_limits![limit] = value;
                     }
                     break;
                  case 'host_limits':
                     for (const [limit, value] of Object.entries(project.host_limits!)) {
                        createdProject.host_limits![limit] = value;
                     }
                     break;

                  case '_owners':
                     createdProject._owners = createProjectOwners({
                        actualOwners: createdProject.owners,
                     });
                     break;

                  default:
                     createdProject[field] = project[field];
                     break;
               }
            }

            const [projectSavingObs] = this.save(original, createdProject, project.reason ?? '');

            return projectSavingObs.pipe(switchMap(() => this.getById(original.id)));
         }),
      );
   }

   /**
    * Обновляет информацию о проекте.
    *
    * Очень сложный метод, т.к. выполняет не один запрос, а много, в зависимости от того, какие поля были изменены.
    *
    * @param original Оригинальный, неизмененный проект (для сравнения)
    * @param edited Измененный проект
    * @param reason Причина внесения изменений (будет продублирована в каждый запрос)
    */
   public save(original: IProject, edited: IProject, reason: string): [Observable<any>, Observable<any[]> | null] {
      const requests: Observable<any>[] = [
         of(null), // Фейковый запрос, чтобы при отсутствии изменений вернуть хоть что-нибудь
      ];

      const directFields: (keyof IProject)[] = [
         'certificate_deploy',
         'default_host_restrictions',
         'manually_disabled_checks',
         'name',
         'tags',
         'yc_iam_folder_id',

         // cauth
         'cauth_flow_type',
         'cauth_insecure_ca_list_url',
         'cauth_key_sources',
         'cauth_krl_url',
         'cauth_secure_ca_list_url',
         'cauth_sudo_ca_list_url',
         'cauth_trusted_sources',
      ];
      const directPatch = _getPatchForFields(original, edited, directFields);
      if (!isEmpty(directPatch)) {
         requests.push(this.request(HttpMethod.PATCH, original.id, null, { ...directPatch, reason }));
      }

      // Только админы могут управлять bot_project_id
      const botProjectIdPatch = _getPatchForFields(original, edited, ['bot_project_id']);
      if (!isEmpty(botProjectIdPatch)) {
         requests.push(
            this.request(HttpMethod.POST, `${original.id}/bot_project_id`, null, { ...botProjectIdPatch, reason }),
         );
      }

      const cmsPath = _getPatchForFields(original, edited, ['cms']);
      if (!isEmpty(cmsPath)) {
         requests.push(
            this.request(HttpMethod.PATCH, `${original.id}/cms_settings`, null, {
               cms_settings: cmsPath.cms,
               reason,
            }),
         );
      }

      const notificationsPatch = _getPatchForFields(original, edited, ['notifications']);
      if (!isEmpty(notificationsPatch)) {
         requests.push(
            this.request(HttpMethod.PUT, `${original.id}/notifications/recipients`, null, {
               ...notificationsPatch.notifications!.recipients,
               reason,
            }),
         );
      }

      // Только админы могут управлять dns_domain
      const dnsDomainPatch = _getPatchForFields(original, edited, ['dns_domain']);
      if (!isEmpty(dnsDomainPatch)) {
         requests.push(this.updateDnsDomain(original.id, dnsDomainPatch.dns_domain!, reason));
      }

      const reportsPatch = _getPatchForFields(original, edited, ['reports']);
      if (!isEmpty(reportsPatch)) {
         const clearedReports = clearReportsParams(reportsPatch!.reports!);
         requests.push(this.request(HttpMethod.POST, `${original.id}/reports`, null, { ...clearedReports, reason }));
      }

      const automationLimitsPatch = _getPatchForFields(original, edited, ['automation_limits']);
      if (!isEmpty(automationLimitsPatch)) {
         requests.push(
            this.request(HttpMethod.POST, `${original.id}/automation_limits`, null, {
               ..._cleanDuplicatedLimits(automationLimitsPatch.automation_limits!),
               reason,
            }),
         );
      }

      const hostLimitsPatch = _getPatchForFields(original, edited, ['host_limits']);
      if (!isEmpty(hostLimitsPatch)) {
         requests.push(
            this.request(HttpMethod.POST, `${original.id}/host_limits`, null, {
               ..._cleanDuplicatedLimits(hostLimitsPatch.host_limits!),
               reason,
            }),
         );
      }

      const maintenancePlotPatch = _getPatchForFields(original, edited, ['maintenance_plot_id']);
      if (!isEmpty(maintenancePlotPatch)) {
         requests.push(
            maintenancePlotPatch.maintenance_plot_id
               ? this.request(HttpMethod.POST, `${original.id}/maintenance_plot`, null, {
                    maintenance_plot_id: maintenancePlotPatch.maintenance_plot_id,
                    reason,
                 })
               : this.request(HttpMethod.DELETE, `${original.id}/maintenance_plot`, null, { reason }),
         );
      }

      const automationPlotPatch = _getPatchForFields(original, edited, ['automation_plot_id']);
      if (!isEmpty(automationPlotPatch)) {
         requests.push(
            automationPlotPatch.automation_plot_id
               ? this.request(HttpMethod.POST, `${original.id}/automation_plot`, null, {
                    automation_plot_id: automationPlotPatch.automation_plot_id,
                    reason,
                 })
               : this.request(HttpMethod.DELETE, `${original.id}/automation_plot`, null, { reason }),
         );
      }

      const profilePatch = _getPatchForFields(original, edited, ['profile', 'profile_tags']);
      if (!isEmpty(profilePatch)) {
         requests.push(
            this.request(HttpMethod.PUT, `${original.id}/host-profiling-config`, null, {
               profile: edited.profile,
               profile_tags: edited.profile_tags,
               reason,
            }),
         );
      }

      const deployFields: (keyof IProject)[] = [
         'deploy_config',
         'deploy_config_policy',
         'deploy_network',
         'deploy_tags',
         'provisioner',
      ];
      const deployPatch = _getPatchForFields(original, edited, deployFields);
      if (!isEmpty(deployPatch)) {
         requests.push(
            this.request(HttpMethod.PUT, `${original.id}/host-provisioner-config`, null, {
               deploy_config: edited.deploy_config,
               deploy_config_policy: edited.deploy_config_policy,
               deploy_network: edited.deploy_network,
               deploy_tags: edited.deploy_tags,
               provisioner: edited.provisioner,
               reason,
            }),
         );
      }

      const vlanFields: (keyof IProject)[] = [
         'extra_vlans',
         'hbf_project_id',
         'ip_method',
         'native_vlan',
         'vlan_scheme',
         'use_fastbone',
      ];
      const vlanPatch = _getPatchForFields(original, edited, vlanFields);
      if (!isEmpty(vlanPatch)) {
         switch (edited.vlan_scheme) {
            case VlanSchema.MTN:
               requests.push(
                  this.request(HttpMethod.PATCH, `${original.id}/hbf_project_id`, null, {
                     hbf_project_id: edited.hbf_project_id,
                     ip_method: edited.ip_method,
                     use_fastbone: edited.use_fastbone,
                     reason,
                  }),
               );
               break;

            case VlanSchema.None:
               if (original.vlan_scheme) {
                  requests.push(this.request(HttpMethod.DELETE, `${original.id}/vlan_scheme`, null, { reason }));
               }
               break;

            default:
               requests.push(
                  this.request(HttpMethod.PUT, `${original.id}/vlan_scheme`, null, {
                     extra_vlans: edited.extra_vlans,
                     native_vlan: edited.native_vlan,
                     reason,
                     scheme: edited.vlan_scheme,
                  }),
               );
         }
      }

      const ownedVlansPatch = _getPatchForFields(original, edited, ['owned_vlans']);
      if (!isEmpty(ownedVlansPatch)) {
         requests.push(
            this.request(HttpMethod.PUT, `${original.id}/owned_vlans`, null, {
               reason,
               vlans: ownedVlansPatch.owned_vlans,
            }),
         );
      }

      const othersRebootViaSsh = _getPatchForFields(original, edited, ['reboot_via_ssh']);
      if (!isEmpty(othersRebootViaSsh)) {
         requests.push(
            othersRebootViaSsh.reboot_via_ssh
               ? this.request(HttpMethod.PUT, `${original.id}/rebooting_via_ssh`, null, { reason })
               : this.request(HttpMethod.DELETE, `${original.id}/rebooting_via_ssh`, null, { reason }),
         );
      }

      const othersTier = _getPatchForFields(original, edited, ['tier']);
      if (!isEmpty(othersTier)) {
         requests.push(this.request(HttpMethod.PATCH, `${original.id}/tier`, null, { reason, tier: edited.tier }));
      }

      return [forkJoin(requests), this._saveOwners(original, edited, reason)];
   }

   public clone(from: string, params: ICloneProjectParams): Observable<IProject> {
      return this.request<void, ICloneProjectParams, IProject>(
         HttpMethod.POST,
         `clone/${from}`,
         undefined,
         params,
      ).pipe(tap(_fillProjectFields));
   }

   public remove(id: string, params: IProjectActionParams): Observable<void> {
      return this.request(HttpMethod.DELETE, id, null, params);
   }

   public getRequestedOwners(projectId: string): Observable<Map<string, IdmRequest>> {
      return this.request<void, IdmRequest, { result: Record<string, number> }>(
         HttpMethod.GET,
         `${projectId}/requested_owners_with_request_id`,
      ).pipe(map(r => new Map(Object.keys(r.result).map(login => [login, { id: r.result[login], login }]))));
   }

   public getRevokingOwners(projectId: string): Observable<Map<string, IdmRequest>> {
      return this.request<void, IdmRequest, { result: Record<string, number> }>(
         HttpMethod.GET,
         `${projectId}/revoking_owners_with_request_id`,
      ).pipe(map(r => new Map(Object.keys(r.result).map(login => [login, { id: r.result[login], login }]))));
   }

   public enableAutomation(
      type: AutomationType,
      projectId: string,
      params: Partial<IAutomationCreditsForm>,
   ): Observable<void> {
      return this.request(HttpMethod.PUT, `${projectId}/enable_automation/${type}`, null, params);
   }

   public disableAutomation(type: AutomationType, projectId: string, reason: string): Observable<void> {
      return this.request(HttpMethod.DELETE, `${projectId}/enable_automation/${type}`, null, { reason });
   }

   public enableFsmHandbrake(projectId: string, params: IFsmHandbrakeProjectParams) {
      switch (params._mode) {
         case TimeoutMode.Date: {
            delete params.timeout;

            if (params.timeout_time instanceof Date) {
               params.timeout_time = toTimestamp(params.timeout_time);
            }
            break;
         }

         case TimeoutMode.Timer: {
            delete params.timeout_time;
            break;
         }
      }

      delete params._mode;

      return this.request<void, IFsmHandbrakeProjectParams, IProject>(
         HttpMethod.POST,
         `${projectId}/fsm-handbrake`,
         undefined,
         params,
      );
   }

   public disableFsmHandbrake(projectId: string, reason: string): Observable<void> {
      return this.request(HttpMethod.DELETE, `${projectId}/fsm-handbrake`, null, { reason });
   }

   public updateDnsDomain(projectId: string, dnsDomain: string, reason: string, dnsZoneId?: string): Observable<void> {
      return isEmpty(dnsDomain)
         ? this.request(HttpMethod.DELETE, `${projectId}/dns_domain`, null, { reason })
         : this.request(HttpMethod.POST, `${projectId}/dns_domain`, null, {
              dns_domain: dnsDomain,
              yc_dns_zone_id: dnsZoneId,
              reason,
           });
   }

   public addRoleMember(projectId: string, role: string, member: string): Observable<string> {
      const body = { member };

      return this.request(HttpMethod.POST, `${projectId}/role/${role}/members`, undefined, body).pipe(
         map(() => member),
      );
   }

   public removeRoleMember(projectId: string, role: string, member: string): Observable<string> {
      const body = { member };

      return this.request(HttpMethod.DELETE, `${projectId}/role/${role}/members`, undefined, body).pipe(
         map(() => member),
      );
   }

   private _saveOwners(original: IProject, edited: IProject, reason: string): Observable<any[]> | null {
      const ownersPatch = _getPatchForFields(original, edited, ['_owners']);
      if (isEmpty(ownersPatch)) {
         return null;
      }

      const ownersRequests: Observable<void>[] = [];

      if (ownersPatch._owners!.toAdd.size > 0) {
         ownersRequests.push(
            this.request(HttpMethod.POST, `${original.id}/owners`, null, {
               owners: ownersPatch._owners!.toAdd,
               reason,
            }),
         );
      }

      if (ownersPatch._owners!.toRemove.size > 0) {
         ownersRequests.push(
            this.request(HttpMethod.DELETE, `${original.id}/owners`, null, {
               owners: ownersPatch._owners!.toRemove,
               reason,
            }),
         );
      }

      return forkJoin(ownersRequests);
   }
}

function _getPatchForFields<T>(original: T, edited: T, fields: (keyof T)[]): Partial<T> {
   const result: Partial<T> = {};

   for (const field of fields) {
      if (!isEqual(original[field], edited[field])) {
         result[field] = edited[field];
      }
   }

   return result;
}

function _fillProjectFields(project: IProject): void {
   if (project.automation_limits) {
      project.automation_limits = _convertAutomationLimits(project.automation_limits || {});
   }

   if (project.host_limits) {
      project.host_limits = _convertAutomationLimits(project.host_limits || {});
   }

   fillSetFields(project, [
      'deploy_tags',
      'extra_vlans',
      'owned_vlans',
      'owners',
      'profile_tags',
      'default_host_restrictions',
      'manually_disabled_checks',
      // 'tags',
   ]);

   if (!isEmpty(project?.cms_settings?.[0])) {
      project.cms = _convertCmsField(project.cms_settings! as ProjectCmsSettings[]);
      delete project.cms_settings;
   }

   if (project.provisioner === Provisioner.LUI && isEmpty(project.deploy_network)) {
      project.deploy_network = DefaultDeployNetwork;
   }

   // Жуткий костыль, но так было в старом UI и обусловлено общей костыльностью обработки mtn-hostid схемы в API
   // https://st.yandex-team.ru/WALLEUI-658
   if (project.vlan_scheme) {
      if (project.vlan_scheme === VlanSchema.MTNHostId) {
         project.vlan_scheme = VlanSchema.MTN;
         project.ip_method = 'hostname';
      } else if (project.vlan_scheme === VlanSchema.MTN) {
         project.ip_method = 'mac';
      }
   }

   if (!project.reports) {
      project.reports = {
         enabled: false,
         extra: {},
         queue: '',
         summary: '',
      };
   }
}

function _splitProjectAutomationLimits(project: IProject, constants: IConstants) {
   if (!project.automation_limits) {
      return;
   }

   project._checks = {};

   const systemLimits = new Set(constants.project_automation_limits!);
   const projectLimits = new Set(Object.keys(project.automation_limits!));
   const { added: customLimits } = getSetDifference(systemLimits, projectLimits);
   customLimits.forEach(limitKey => {
      project._checks![limitKey] = project.automation_limits![limitKey];
      delete project.automation_limits![limitKey];
   });
}

// Косяк в API, структура на входе отличается от структуры на выходе.
// Привожу к единой структуре (к той, что на вход, в виде отдельного объекта)
// экспортируется для storybook
export function _convertCmsField(cmsSettings: ProjectCmsSettings[]): CmsValue[] {
   return cmsSettings.map(el => {
      if (el.cms === 'default') {
         return {
            max_busy_hosts: el.cms_max_busy_hosts,
            url: 'default',
         } as IDefaultCms;
      }

      return {
         api_version: el.cms_api_version,
         tvm_app_id: el.cms_tvm_app_id,
         temporary_unreachable_enabled: el?.temporary_unreachable_enabled,
         url: el.cms as string,
      } as ICustomCms;
   });
}

// экспорт для storybook
export function _convertAutomationLimits(limits: IAutomationLimits): IAutomationLimits {
   const result: IAutomationLimits = {};

   interface IRawLimitRecord {
      limit: number;
      period: string;
   }

   for (const key in limits) {
      if (limits.hasOwnProperty(key)) {
         const before: IRawLimitRecord[] = limits[key] as any[];
         result[key] = uniqueBy(before, l => `${l.limit}|${l.period}`).map(l => new AutomationLimit(l.limit, l.period));
      }
   }

   return result;
}

// экспорт для тестов
export function _cleanDuplicatedLimits(limits: IAutomationLimits): IAutomationLimits {
   const result: IAutomationLimits = {};

   for (const key in limits) {
      if (limits.hasOwnProperty(key)) {
         const before = limits[key];
         result[key] = uniqueBy(before, l => `${l.times}|${l.getFullPeriod()}`);
      }
   }

   return result;
}

export const projectApi = new ProjectApi(`${config.walleApi}/v1/projects`);
