import { HttpMethod } from '@yandex-infracloud-ui/libs-next';
import { Observable } from 'rxjs';

import {
   IListResult,
   IScenario,
   IScenarioCancelParams,
   IScenarioFilters,
   IScenarioFinishParams,
   IScenarioListRequest,
   IScenarioPauseParams,
   IScenarioSkipParams,
   IScenarioStartParams,
   ScenarioParams,
} from '../../models';
import { config } from '../config';
import { parseHostQuery } from './api_helpers';
import { WalleBaseApi } from './base_api';

const allScenariosFields: (keyof IScenario)[] = [
   'action_time',
   'creation_time',
   'data_storage',
   'errors',
   'message',
   'hosts',
   'issuer',
   'labels',
   'name',
   'scenario_id',
   'scenario_type',
   'script_args',
   'status',
   'ticket_key',
];

const shortScenariosFields = allScenariosFields.filter(f => f !== 'hosts' && f !== 'message');

class ScenariosApi extends WalleBaseApi {
   public getList(limit: number, offset: number, filters: IScenarioFilters): Observable<IListResult<IScenario>> {
      const { invs: ids, names } = parseHostQuery(filters.query); // TODO parseScenarioQuery

      const params: Partial<IScenarioListRequest> = {
         fields: shortScenariosFields,
         issuer: filters.issuer,
         work_status: filters.workStatus,
         scenario_type: filters.type,
         limit,
         name: new Set(names),
         offset,
         scenario_id: new Set(ids),
      };

      return this.request<Partial<IScenarioListRequest>, void, IListResult<IScenario>>(
         HttpMethod.GET,
         'scenarios',
         params,
      );
   }

   public getById(id: string): Observable<IScenario> {
      return this.request<Partial<IScenarioListRequest>, void, IScenario>(HttpMethod.GET, `scenarios/${id}`, {
         fields: allScenariosFields,
      });
   }

   public getStateById(id: string): Observable<any> {
      return this.request<Partial<any>, void, any>(HttpMethod.GET, `scenarios/${id}/snapshot`);
   }

   public createHostsTransfer(scenario: Partial<ScenarioParams>): Observable<IScenario> {
      return this.request(HttpMethod.POST, 'scenarios', null, scenario);
   }

   public createReservedHostsTransfer(scenario: Partial<ScenarioParams>): Observable<IScenario> {
      return this.request(HttpMethod.POST, 'scenarios', null, scenario);
   }

   public createItdcMaintenance(scenario: Partial<ScenarioParams>): Observable<IScenario> {
      return this.request(HttpMethod.POST, 'scenarios', null, scenario);
   }

   public createNocHard(scenario: Partial<ScenarioParams>): Observable<IScenario> {
      return this.request(HttpMethod.POST, 'scenarios', null, scenario);
   }

   public createNocSoft(scenario: Partial<ScenarioParams>): Observable<IScenario> {
      return this.request(HttpMethod.POST, 'scenarios', null, scenario);
   }

   public start(id: number, params: IScenarioStartParams): Observable<void> {
      return this.request(HttpMethod.PATCH, `scenarios/${id}/start`, null, params);
   }

   public pause(id: number, params: IScenarioPauseParams): Observable<void> {
      return this.request(HttpMethod.PATCH, `scenarios/${id}/pause`, null, params);
   }

   public skip(id: number, params: IScenarioSkipParams): Observable<void> {
      return this.request(HttpMethod.POST, `scenarios/${id}/apply-action`, null, {
         'action': 'skip_wait_stage_for_hosts_transfer',
         ...params,
      });
   }

   public finish(id: number, params: IScenarioFinishParams): Observable<void> {
      return this.request(HttpMethod.PATCH, `scenarios/${id}`, null, {
         'labels': { 'WORK_COMPLETED': 'true' },
         reason: params.reason,
      });
   }

   public cancel(id: number, params: IScenarioCancelParams): Observable<void> {
      return this.request(HttpMethod.PATCH, `scenarios/${id}/cancel`, null, params);
   }
}

export const scenariosApi = new ScenariosApi(`${config.walleApi}/v1`);
