import { toasts } from '@yandex-infracloud-ui/libs';
import { filter, tap } from 'rxjs/operators';
import { config } from 'services';

import {
   EXTERNAL_LINKS,
   globalState,
   IAutomationPlot,
   IHost,
   IMaintenancePlot,
   IScenario,
   IUserInfo,
   ProjectType,
   UserModel,
} from '../models';

import { dictApi } from './api/dict_api';
import { yandexCookies } from './yandexCookies';

class Auth {
   public canEditScenario = (scenario: IScenario | null, user = globalState.user.value): boolean => {
      if (!user || !scenario) {
         return false;
      }

      if (user.isAdmin) {
         return true;
      }

      const login = user.login.replace(/@$/, '');

      const issuer = scenario.issuer.replace(/@$/, '');
      const responsible = scenario.labels?.responsible;
      const ticketAuthor = scenario.labels?.ticket_created_by;

      return new Set([issuer, responsible, ticketAuthor]).has(login);
   };

   public init() {
      // Изначально берём пользователя из кук, если есть
      const uid = yandexCookies.getYandexUID();
      if (!uid) {
         this._goToLogin();

         return;
      }

      const login = config.isExternal ? 'robot-walle' : yandexCookies.getYandexLogin(); // external installation does not support cookie login
      const user = new UserModel(uid, login, login, false, 'cookie', [], [], {}, []);

      globalState.user.next(user);

      // Сервисы API при 401 ошибке очищают user
      globalState.user
         .pipe(
            filter(u => u === null),
            tap(() => globalState.csrfToken.next('')),
         )
         .subscribe(() => this._goToLogin());

      // загрузка данных для WALL-E
      this._load();
      this._loadCSRF();
   }

   public canEditHost(host: IHost | null, user = globalState.user.value): boolean {
      if (!user || !host) {
         return false;
      }

      if (user.isAdmin) {
         return true;
      }

      return user.projects.includes(host.project);
   }

   public canRunHostAction(host: IHost | null, user = globalState.user.value): boolean {
      if (!user || !host || host.type === ProjectType.SHADOW) {
         return false;
      }

      if (user.isAdmin) {
         return true;
      }

      return (
         user.projectRoles[host.project]?.some(role => role === 'owner' || role === 'user' || role === 'superuser') ??
         false
      );
   }

   public canEditProject(project: { id: string; owners?: Set<string> } | null, user = globalState.user.value): boolean {
      if (!user || !project) {
         return false;
      }

      if (user.isAdmin || config.isExternal) {
         return true;
      }

      return this._isOwner(user, project.id, project.owners, u => u.projects);
   }

   public canEditBotProject(user = globalState.user.value): boolean {
      if (!user) {
         return false;
      }

      return config.isExternal || user.isAdmin;
   }

   public canEditDnsDomain(user = globalState.user.value): boolean {
      if (!user) {
         return false;
      }

      return config.isExternal || user.isAdmin;
   }

   public canEditPlot(plot: IAutomationPlot | IMaintenancePlot | null, user = globalState.user.value): boolean {
      if (!user || !plot) {
         return false;
      }

      if (user.isAdmin || config.isExternal) {
         return true;
      }

      return this._isOwner(
         user,
         plot.id,
         // для maintenance plot'ов меняю формат slug'а с abc на staff
         // tslint:disable-next-line:no-string-literal
         plot?.['owners'] ?? new Set([`@svc_${plot?.['meta_info']?.['abc_service_slug']}`]),
         u => u.automationPlots,
      );
   }

   public canEditOwnedVlans(user = globalState.user.value): boolean {
      return config.isExternal || (user ? user.isAdmin : false);
   }

   private _goToLogin() {
      // external installation is redirected on the server
      if (config.isExternal) {
         return;
      }
      window.location.href = EXTERNAL_LINKS.passport(window.location.href);
   }

   private _isOwner(
      user: UserModel,
      id: string,
      owners: Set<string> | null | undefined,
      extractor: (user: UserModel) => string[],
   ): boolean {
      // Пользователь прямой владелец
      if (owners && owners.has(user.login)) {
         return true;
      }

      // Пользователь владелец в составе группы
      if (owners && user.groups && user.groups.some(g => owners.has(g))) {
         return true;
      }

      // В модели пользователя указана такая сущность
      return extractor(user).includes(id);
   }

   private _load() {
      dictApi.getCurrentUser().subscribe((apiUser: IUserInfo) => {
         const userId = yandexCookies.getYandexUID();
         const user = new UserModel(
            userId,
            apiUser.login,
            apiUser.login,
            apiUser.admin,
            'api',
            apiUser.groups,
            apiUser.projects,
            apiUser.project_roles,
            apiUser.automation_plots,
         );

         globalState.user.next(user);
      }, toasts.handleApiError('Current user loading'));
   }

   private _loadCSRF() {
      dictApi
         .getCsrfToken()
         .subscribe(token => globalState.csrfToken.next(token), toasts.handleApiError('CSRF token loading'));
   }
}

export const auth = new Auth();
