import { NullableDate } from '@yandex-infracloud-ui/libs';
import { DEFAULT_THEME, DefaultOwnership, OwnershipValue, Theme } from '../models';

export const WALLE_API_URLS = {
   mock: '//localhost.yandex-team.ru:3002',
   test: '//api.wall-e-test.yandex-team.ru',
   prod: '//api.wall-e.yandex-team.ru',
};

export const WALLE_INSTALLATION_TYPES = {
   internal: 'internal',
   external: 'external',
};

export type ApiPreset = keyof typeof WALLE_API_URLS;

export type InstallationPreset = keyof typeof WALLE_INSTALLATION_TYPES;

const DefaultApiPreset: ApiPreset = 'mock';

const DefaultInstallationPreset: InstallationPreset = 'internal';

const DEFAULT_UI_CONFIG: IUiConfig = {
   feedbackFormId: 22776,
   idmStand: 'idm',
   idmSystemId: 'walle-testing',
   infraBuzzer: false, // {serviceId: 105, environmentId: 208}
   metrika: false,
   profiling: false,
   sentryDsn: 'https://56175014d1c346b7a49f29212d932ae3@sentry.iddqd.yandex.net/719',
   staffApi: '//staff-api.yandex-team.ru/v3/',
   installationType: WALLE_INSTALLATION_TYPES.internal,
   startrekApi: '//st-api.yandex-team.ru/v2/',
   walleApi: WALLE_API_URLS.mock,
};

// TODO переименовать в Environment
class Config {
   private readonly _config: IUiConfig;

   private readonly _now: NullableDate;

   private readonly _version: string;

   private apiPresetStorageKey = 'settings.api.preset';

   private installationPresetStorageKey = 'settings.installation.preset';

   private defaultOwnershipStorageKey = 'state.projects.defaultOwnership';

   private isProjectListCollapsedStorageKey = 'state.projects.isCollapsed';

   constructor(_env = process.env) {
      this._now = _env.REACT_APP_NOW ? new Date(_env.REACT_APP_NOW) : null;
      this._config = { ...DEFAULT_UI_CONFIG, ...window.UI_CONFIG };
      this._version = _env.REACT_APP_VERSION as string;
   }

   public get isLocalhost(): boolean {
      return window.location.host.includes('localhost');
   }

   public get isExternal(): boolean {
      return this.installationType === WALLE_INSTALLATION_TYPES.external;
   }

   public get installationType(): InstallationPreset {
      return this._getFromStorage(
         this.installationPresetStorageKey,
         this._config.installationType ?? DefaultInstallationPreset,
      ) as InstallationPreset;
   }

   public set installationType(v: InstallationPreset) {
      this._saveToStorage(this.installationPresetStorageKey, v);
   }

   public get isProjectListCollapsed(): boolean {
      return this._getFromStorage(this.isProjectListCollapsedStorageKey, false);
   }

   public set isProjectListCollapsed(v: boolean) {
      this._saveToStorage(this.isProjectListCollapsedStorageKey, v);
   }

   public get ownershipDefault(): OwnershipValue {
      return this._getFromStorage(this.defaultOwnershipStorageKey, DefaultOwnership);
   }

   public set ownershipDefault(v: OwnershipValue) {
      this._saveToStorage(this.defaultOwnershipStorageKey, v);
   }

   public get enabledDev(): boolean {
      return !('disableDev' in localStorage);
   }

   public get rawConfig() {
      return this._config;
   }

   public get now() {
      return this._now;
   }

   public get walleApi(): string {
      if (this.isLocalhost) {
         return WALLE_API_URLS[this.apiPreset];
      }

      return this._config.walleApi;
   }

   public get startrekApi(): string {
      return this._config.startrekApi;
   }

   public get apiPreset(): ApiPreset {
      return this._getFromStorage(this.apiPresetStorageKey, DefaultApiPreset) as ApiPreset;
   }

   public set apiPreset(v: ApiPreset) {
      this._saveToStorage(this.apiPresetStorageKey, v);
   }

   /**
    * Для откладки производительности
    */
   public get profiling(): boolean {
      return this._config.profiling;
   }

   public get staffApi(): string {
      return this._config.staffApi;
   }

   public get useInfraBuzzer(): boolean {
      return this.infraBuzzer.length > 0;
   }

   public get feedbackFormId(): number {
      return this._config.feedbackFormId;
   }

   public get infraBuzzer(): IInfraBuzzerService[] {
      if (Array.isArray(this._config.infraBuzzer)) {
         return this._config.infraBuzzer;
      }

      if (this._config.infraBuzzer) {
         return [this._config.infraBuzzer];
      }

      return [];
   }

   public get metrikaId(): number | null {
      return this._config.metrika !== false ? this._config.metrika : null;
   }

   public get theme(): Theme {
      return this._getFromStorage('ui.theme', DEFAULT_THEME);
   }

   public set theme(v) {
      this._saveToStorage('ui.theme', v);
   }

   public get sentryDsn(): string | null {
      return this._config.sentryDsn !== false ? this._config.sentryDsn : null;
   }

   public get version(): string {
      return this._version;
   }

   /**
    * Возвращает ссылку на уже заполненную форму заявки для добавления нового овнера в проект
    */
   public getIdmAddOwnerUrl(projectId: string, login: string) {
      const roleRandomId = Math.random().toFixed(10).slice(2);

      const idmSystemId = this._config.idmSystemId;

      // noinspection LongLine
      return `${this._getIdmWalleSystem()}/roles#rf=1,rf-role=${roleRandomId}#user:${login}@${idmSystemId}/project/${projectId}/owner(fields:();params:()),f-status=all,f-role=${idmSystemId},sort-by=-updated,f-is-expanded=true,rf-expanded=${roleRandomId}`;
   }

   /**
    * Возвращает ссылку на запрос роли в IDM по конкретному реквесту
    */
   public getIdmRequestOwnerUrl(requestId: number): string {
      return `${this._getIdmWalleSystem()}#role=${requestId},f-role-id=${requestId}`;
   }

   /**
    * Возвращает ссылку на запрос доступа к хосту
    */
   public getIdmRequestHostAccessUrl(login: string, hostName: string): string {
      // noinspection LongLine
      return `https://idm.yandex-team.ru/user/${login}/roles#rf=1,rf-role=HNyKmjk4#user:${login}@cauth/${hostName}(fields:();params:()),rf-expanded=HNyKmjk4,f-status=all,sort-by=-updated`;
   }

   /**
    * Для работы с local storage
    */
   private _getFromStorage<T>(key: string, defaultValue: T): T {
      const fromStorage = localStorage.getItem(key);

      return fromStorage ? JSON.parse(fromStorage) : defaultValue;
   }

   private _getIdmWalleSystem(): string {
      const idmSystemId = this._config.idmSystemId;
      const idmStand = this._config.idmStand;

      return `https://${idmStand}.yandex-team.ru/system/${idmSystemId}`;
   }

   private _saveToStorage<T>(key: string, v: T) {
      localStorage.setItem(key, JSON.stringify(v));
   }
}

export const config = new Config();
