import {
   Breadcrumb,
   captureException,
   captureMessage,
   configureScope,
   init as initSentry,
   Severity,
   withScope,
} from '@sentry/browser';
import { IApiError, toasts, ToastType } from '@yandex-infracloud-ui/libs';
import { filter, map } from 'rxjs/operators';

import { globalState } from '../models';
import { config } from './config';

interface ISentryMessage {
   message: string;
   severity: Severity;
   tags: any;
}

function _errorTolerantToJson(error: IApiError) {
   try {
      return JSON.stringify(error);
   } catch (e) {
      return `${error}`;
   }
}

class Sentry {
   private _blacklistWords = [
      '@bem/sdk', // Из недр LEGO
      'dropdown2', // Из недр LEGO
      'popup2', // Из недр LEGO
   ];

   private delayedMessages: ISentryMessage[] = [];

   public init() {
      if (!config.sentryDsn || config.isLocalhost) {
         return;
      }

      initSentry({
         beforeBreadcrumb: (breadcrumb: Breadcrumb): Breadcrumb | null => {
            if (breadcrumb.message) {
               for (const blacklistWord of this._blacklistWords) {
                  if (breadcrumb.message.includes(blacklistWord)) {
                     return null;
                  }
               }
            }

            return breadcrumb;
         },
         dsn: config.sentryDsn,
         release: config.version,
      });

      globalState.user.subscribe(user => {
         // pre-configuration
         configureScope(scope => {
            scope.setUser({
               id: user ? user.login : 'anonymous',
               username: user ? user.name : 'anonymous',
            });
         });

         // send all delayed messages
         this.delayedMessages.forEach(message => this.directSend(message));
         this.delayedMessages = [];
      });

      toasts.toasts
         .pipe(
            filter(toast => toast.type === ToastType.Error),
            map(toast => ({ title: toast.title, text: toast.text })),
         )
         .subscribe(toastInfo =>
            this.send({
               message: `Toast: ${toastInfo.title ? toastInfo.title + ': ' : ''}${toastInfo.text}`,
               severity: Severity.Warning,
               tags: { text: toastInfo.text },
            }),
         );
   }

   public sendApiError(resp: Response, error: IApiError) {
      this.send({
         message: `API Error: ${error.errorMessage || _errorTolerantToJson(error)}`,
         severity: Severity.Error,
         tags: { status: resp.status },
      });
   }

   public sendRedirect(referrer: string, oldRoute: string, newRoute: string) {
      this.send({
         message: `Redirect from old route: ${referrer}`,
         severity: Severity.Info,
         tags: { oldRoute, newRoute },
      });
   }

   public sendJsError(error: Error) {
      captureException(error);
   }

   private directSend(message: ISentryMessage) {
      withScope(scope => {
         Object.entries(message.tags).forEach(([tag, value]) => {
            scope.setTag(tag, `${value}`);
         });

         captureMessage(message.message, message.severity);
      });
   }

   private send(message: ISentryMessage) {
      if (globalState.user.getValue()) {
         this.directSend(message);
      } else {
         this.delayedMessages.push(message);
      }
   }
}

export const sentry = new Sentry();
