import { EMPTY_VALUE, Loader, toasts, useDismounted, useDropdownDimensions } from '@yandex-infracloud-ui/libs';
import { Popup, TextInput } from 'lego-on-react';
import * as React from 'react';
import { SyntheticEvent, useCallback, useEffect, useReducer, useRef } from 'react';
import { takeUntil } from 'rxjs/operators';

import { IDeployConfig } from '../../models';
import { dictApi } from '../../services';

import { DeployConfigList } from './DeployConfigList';
import styles from './DeployConfigSelect.module.css';
import { ActionType, initialState, reducer } from './state';

interface IProps {
   cls?: string;
   disabled?: boolean;
   initialQuery?: string;
   name?: string;

   onSelect(e: SyntheticEvent | null, value: IDeployConfig | null): void;

   onBlur?(): void;
}

export const DeployConfigSelect = React.memo(
   ({ cls = '', initialQuery = '', disabled = false, name, onBlur, onSelect }: IProps) => {
      // hooks
      const firstRun = useRef(true);
      const dismounted = useDismounted();
      const switcherRef = useRef<HTMLElement>();
      const [, maxHeight] = useDropdownDimensions(switcherRef);
      const [state, dispatch] = useReducer(reducer, initialState);

      // helpers
      const _loadConfigs = useCallback(() => {
         dispatch({ type: ActionType.BeforeConfigLoading });

         dictApi
            .getDeployConfigs()
            .pipe(takeUntil(dismounted))
            .subscribe(
               configs => dispatch({ type: ActionType.ConfigLoaded, configs }),
               resp => {
                  dispatch({ type: ActionType.ConfigLoadError });
                  toasts.apiError('Deploy configs loading', resp);
               },
            );
      }, [dismounted]);

      // effects
      useEffect(() => {
         if (firstRun.current && initialQuery) {
            dispatch({ type: ActionType.JustSetQuery, query: initialQuery });
            firstRun.current = false;
         }
      }, [initialQuery]);

      // handlers
      const onWrite = useCallback(
         (q: string) => {
            if (q === '') {
               onSelect(null, null);
            }

            dispatch({ type: ActionType.UpdateQuery, query: q });
         },
         [onSelect],
      );

      const open = useCallback(() => {
         dispatch({ type: ActionType.Open });

         if (!state.configs) {
            _loadConfigs();
         }
      }, [_loadConfigs, state.configs]);

      const onItemSelect = useCallback(
         (e: SyntheticEvent, config: IDeployConfig): void => {
            dispatch({ type: ActionType.ItemSelect, config });
            onSelect(null, config);
         },
         [onSelect],
      );

      const close = useCallback(() => dispatch({ type: ActionType.Close }), []);

      const setInnerRef = useCallback((node: HTMLElement) => (switcherRef.current = node), []);

      // render
      if (disabled) {
         return <>{state.query || EMPTY_VALUE}</>;
      }

      return (
         <>
            <TextInput
               theme={'normal'}
               size={'s'}
               placeholder={'Enter deploy config...'}
               cls={cls}
               disabled={disabled}
               id={name}
               name={name}
               text={state.query}
               onChange={onWrite}
               onBlur={onBlur}
               innerRef={setInnerRef}
               onFocus={open}
               hasClear={true}
            />
            <Popup
               theme={'normal'}
               motionless={true}
               target={'anchor'}
               directions={['bottom-left']}
               onOutsideClick={close}
               visible={state.opened}
               anchor={switcherRef.current}
            >
               {state.isLoading ? <Loader cls={styles.loader} text={'Deploy configs loading'} /> : null}

               {state.suggestions && state.wasOpened ? (
                  <div style={{ maxHeight, overflowY: 'auto' }}>
                     <DeployConfigList
                        eine={state.suggestions.eine}
                        lui={state.suggestions.lui}
                        onSelect={onItemSelect}
                     />
                  </div>
               ) : null}
            </Popup>
         </>
      );
   },
);

DeployConfigSelect.displayName = 'DeployConfigSelect';
