import { IDeployConfig, IDeployConfigs } from '../../models';

export enum ActionType {
   BeforeConfigLoading,
   Close,
   ConfigLoaded,
   ConfigLoadError,
   JustSetQuery,
   ItemSelect,
   Open,
   UpdateQuery,
}

interface IActionBeforeConfigLoading {
   type: ActionType.BeforeConfigLoading;
}

interface IActionClose {
   type: ActionType.Close;
}

interface IActionConfigLoaded {
   configs: IDeployConfigs;
   type: ActionType.ConfigLoaded;
}

interface IActionConfigLoadError {
   type: ActionType.ConfigLoadError;
}

interface IActionJustSetQuery {
   query: string;
   type: ActionType.JustSetQuery;
}

interface IActionItemSelect {
   config: IDeployConfig;
   type: ActionType.ItemSelect;
}

interface IActionOpen {
   type: ActionType.Open;
}

interface IActionUpdateQuery {
   query: string;
   type: ActionType.UpdateQuery;
}

type Action =
   | IActionBeforeConfigLoading
   | IActionClose
   | IActionConfigLoaded
   | IActionConfigLoadError
   | IActionJustSetQuery
   | IActionItemSelect
   | IActionOpen
   | IActionUpdateQuery;

export const initialState = {
   configs: null as IDeployConfigs | null,
   isLoading: false,
   opened: false,
   query: '',
   suggestions: null as IDeployConfigs | null,
   wasOpened: false,
};

type State = Readonly<typeof initialState>;

const _filterAndLimitList = (list: string[], query: string): string[] => {
   const result: string[] = [];
   const q = query.toLowerCase().trim();

   for (const item of list) {
      // Ранний выход
      if (result.length >= 25) {
         break;
      }

      if (item.toLowerCase().includes(q)) {
         result.push(item);
      }
   }

   return result;
};

const _getSuggestions = (configs: IDeployConfigs, query: string): IDeployConfigs => ({
   eine: _filterAndLimitList(configs.eine, query),
   lui: _filterAndLimitList(configs.lui, query),
});

export const reducer = (state: State, action: Action): State => {
   switch (action.type) {
      case ActionType.BeforeConfigLoading: {
         return { ...state, isLoading: true };
      }

      case ActionType.Close: {
         return { ...state, opened: false };
      }

      case ActionType.ConfigLoaded: {
         return {
            ...state,
            configs: action.configs,
            isLoading: false,
            suggestions: _getSuggestions(action.configs, state.query),
         };
      }

      case ActionType.ConfigLoadError: {
         return { ...state, isLoading: false };
      }

      case ActionType.JustSetQuery: {
         return {
            ...state,
            query: action.query,
         };
      }

      case ActionType.ItemSelect: {
         return {
            ...state,
            opened: false,
            query: action.config.config,
            suggestions: _getSuggestions(state.configs!, action.config.config),
         };
      }

      case ActionType.Open: {
         return { ...state, opened: true, wasOpened: true };
      }

      case ActionType.UpdateQuery: {
         return {
            ...state,
            opened: true,
            query: action.query,
            suggestions: state.configs ? _getSuggestions(state.configs, action.query) : state.suggestions,
         };
      }

      default: {
         return state;
      }
   }
};
