import { SystemUsersContext } from '@yandex-infracloud-ui/libs';
import { Button } from '@yandex-data-ui/common';
import * as React from 'react';
import { SyntheticEvent, useCallback, useContext, useMemo } from 'react';
// noinspection TypeScriptPreferShortImport,ES6PreferShortImport
import { UserSelect, UserSelectMode } from '../UserSelect/UserSelect';
import classes from './FindUserPopup.module.css';
import { SystemUsersSelector } from './SystemUsersSelector';

interface IProps {
   mode: UserSelectMode;
   submitLabel: string;
   title: string;
   value?: Set<string>;

   /**
    * @param e {SyntheticEvent}
    * @param login Логин пользователя на стаффе, с @ на конце. Например khoden@
    */
   onSubmit(e: SyntheticEvent | null, login: string | null): void;

   onCancel(e: SyntheticEvent): void;
}

export const FindUserPopup: React.FC<IProps> = React.memo(({ mode, title, submitLabel, onCancel, onSubmit, value }) => {
   if (mode === UserSelectMode.Issuers && !value) {
      throw new Error('expected `value` prop if mode is Issuers');
   }

   const systemUsers = useContext(SystemUsersContext);
   const systemUsersSelected = useMemo(
      () => new Set(Array.from<string>(value || new Set()).filter(u => systemUsers.has(u))),
      [systemUsers, value],
   );

   //region handlers
   const handleSubmit = useCallback(
      (e: SyntheticEvent) => {
         e.preventDefault(); // prevent page reloading on form submit

         onCancel(e);
      },
      [onCancel],
   );

   const handleSelect = useCallback(
      (e: SyntheticEvent | null, user: string | null) => {
         if (user !== null) {
            onSubmit(e, user);
         }
      },
      [onSubmit],
   );

   const handleSystemUsersChange = useCallback(
      (selected: Set<string>) => {
         systemUsers.forEach(u => {
            if (systemUsersSelected.has(u.login) !== selected.has(u.login)) {
               onSubmit(null, u.login);
            }
         });
      },
      [onSubmit, systemUsers, systemUsersSelected],
   );
   //endregion

   return (
      <form className={classes.findUserPopup} onSubmit={handleSubmit}>
         <h3>{title}</h3>

         {/*TODO Отключил автофокус, пока не починят Лего*/}
         {/*https://lego-team.slack.com/messages/C0DL13NJY/convo/C0DL13NJY-1543422059.494500/*/}
         <UserSelect mode={mode} onSelect={handleSelect} autofocus={false} />

         {mode === UserSelectMode.Issuers ? (
            <SystemUsersSelector value={systemUsersSelected} onChange={handleSystemUsersChange} />
         ) : null}

         <div className={classes.footer}>
            <Button view={'clear'} size={'m'} onClick={onCancel}>
               Cancel
            </Button>
            <Button view={'action'} size={'m'} type={'submit'}>
               {submitLabel}
            </Button>
         </div>
      </form>
   );
});
