import { HealthStatus, HostMessageSeverity, IHost, IHostHealth, IHostMessages } from '../../models';
import { _extractMessagesBySeverity, _getMessages } from './HostStatusMessages';

describe('HostStatusMessages', () => {
   const messages: IHostMessages = {
      extra: [
         { message: 'aaa', severity: HostMessageSeverity.Info },
         { message: 'bbb', severity: HostMessageSeverity.Error },
         { message: 'ccc', severity: HostMessageSeverity.Error },
      ],
      super: [
         { message: 'xxx', severity: HostMessageSeverity.Error },
         { message: 'yyy', severity: HostMessageSeverity.Info },
      ],
   };

   describe('_extractMessagesBySeverity', () => {
      it('should filter errors', () => {
         expect(_extractMessagesBySeverity(messages, HostMessageSeverity.Error)).toEqual([
            { message: 'extra: bbb', severity: HostMessageSeverity.Error, source: 'Host messages' },
            { message: 'extra: ccc', severity: HostMessageSeverity.Error, source: 'Host messages' },
            { message: 'super: xxx', severity: HostMessageSeverity.Error, source: 'Host messages' },
         ]);
      });

      it('should filter info messages', () => {
         expect(_extractMessagesBySeverity(messages, HostMessageSeverity.Info)).toEqual([
            { message: 'extra: aaa', severity: HostMessageSeverity.Info, source: 'Host messages' },
            { message: 'super: yyy', severity: HostMessageSeverity.Info, source: 'Host messages' },
         ]);
      });
   });

   describe('_getMessages', () => {
      function getFullHost(host: Partial<IHost>): IHost {
         return host as IHost;
      }

      const task = {
         error: 'this is task error',
         status: 'xxx',
         status_message: 'status message',
      };

      const healthOk: IHostHealth = {
         check_statuses: {},
         message: 'health is ok',
         status: HealthStatus.Ok,
      };

      const healthFailure: IHostHealth = {
         check_statuses: {},
         message: 'almost die',
         status: HealthStatus.Failure,
      };

      it("shouldn't throw exception if there aren't any fields", () => {
         expect(_getMessages(getFullHost({}))).toEqual([]);
      });

      it('should return task error and task status message', () => {
         expect(_getMessages(getFullHost({ task }))).toEqual([
            { message: 'this is task error', severity: HostMessageSeverity.Error, source: 'Task error' },
            { message: 'status message', severity: HostMessageSeverity.Info, source: 'Task status message' },
         ]);
      });

      it('should return messages from task.messages', () => {
         expect(_getMessages(getFullHost({ messages }))).toEqual([
            { message: 'extra: bbb', severity: HostMessageSeverity.Error, source: 'Host messages' },
            { message: 'extra: ccc', severity: HostMessageSeverity.Error, source: 'Host messages' },
            { message: 'super: xxx', severity: HostMessageSeverity.Error, source: 'Host messages' },
            { message: 'extra: aaa', severity: HostMessageSeverity.Info, source: 'Host messages' },
            { message: 'super: yyy', severity: HostMessageSeverity.Info, source: 'Host messages' },
         ]);
      });

      it('should return messages from health ok', () => {
         expect(_getMessages(getFullHost({ health: healthOk }))).toEqual([
            { message: 'health is ok', source: 'Host health message', severity: HostMessageSeverity.Info },
         ]);
      });

      it('should return messages from health failure', () => {
         expect(_getMessages(getFullHost({ health: healthFailure }))).toEqual([
            { message: 'almost die', source: 'Host health message', severity: HostMessageSeverity.Info },
         ]);
      });

      it('should return status and state reasons (health ok)', () => {
         expect(_getMessages(getFullHost({ status_reason: 'x', state_reason: 'y', health: healthOk }))).toEqual([
            { message: 'health is ok', source: 'Host health message', severity: HostMessageSeverity.Info },
            { message: 'x', source: 'Status reason', severity: HostMessageSeverity.Info },
            { message: 'y', source: 'State reason', severity: HostMessageSeverity.Info },
         ]);
      });

      it('should return status and state reasons (health failure)', () => {
         expect(_getMessages(getFullHost({ status_reason: 'x', state_reason: 'y', health: healthFailure }))).toEqual([
            { message: 'almost die', source: 'Host health message', severity: HostMessageSeverity.Info },
            { message: 'x', source: 'Status reason', severity: HostMessageSeverity.Info },
            { message: 'y', source: 'State reason', severity: HostMessageSeverity.Info },
         ]);
      });

      it('should return sorted messages with all sources together (health is ok)', () => {
         expect(
            _getMessages(
               getFullHost({
                  health: healthOk,
                  messages,
                  state_reason: 'y',
                  status_reason: 'x',
                  task,
               }),
            ),
         ).toEqual([
            { message: 'this is task error', severity: HostMessageSeverity.Error, source: 'Task error' },
            { message: 'extra: bbb', severity: HostMessageSeverity.Error, source: 'Host messages' },
            { message: 'extra: ccc', severity: HostMessageSeverity.Error, source: 'Host messages' },
            { message: 'super: xxx', severity: HostMessageSeverity.Error, source: 'Host messages' },
            { message: 'extra: aaa', severity: HostMessageSeverity.Info, source: 'Host messages' },
            { message: 'super: yyy', severity: HostMessageSeverity.Info, source: 'Host messages' },
            { message: 'status message', severity: HostMessageSeverity.Info, source: 'Task status message' },
            { message: 'health is ok', source: 'Host health message', severity: HostMessageSeverity.Info },
            { message: 'x', source: 'Status reason', severity: HostMessageSeverity.Info },
            { message: 'y', source: 'State reason', severity: HostMessageSeverity.Info },
         ]);
      });

      it('should return sorted messages with all sources together (health failure)', () => {
         expect(
            _getMessages(
               getFullHost({
                  health: healthFailure,
                  messages,
                  state_reason: 'y',
                  status_reason: 'x',
                  task,
               }),
            ),
         ).toEqual([
            { message: 'this is task error', severity: HostMessageSeverity.Error, source: 'Task error' },
            { message: 'extra: bbb', severity: HostMessageSeverity.Error, source: 'Host messages' },
            { message: 'extra: ccc', severity: HostMessageSeverity.Error, source: 'Host messages' },
            { message: 'super: xxx', severity: HostMessageSeverity.Error, source: 'Host messages' },
            { message: 'extra: aaa', severity: HostMessageSeverity.Info, source: 'Host messages' },
            { message: 'super: yyy', severity: HostMessageSeverity.Info, source: 'Host messages' },
            { message: 'status message', severity: HostMessageSeverity.Info, source: 'Task status message' },
            { message: 'almost die', source: 'Host health message', severity: HostMessageSeverity.Info },
            { message: 'x', source: 'Status reason', severity: HostMessageSeverity.Info },
            { message: 'y', source: 'State reason', severity: HostMessageSeverity.Info },
         ]);
      });
   });
});
