import { EMPTY_VALUE, sortHandler } from '@yandex-infracloud-ui/libs-next';
import * as React from 'react';

import { HostMessageSeverity, IHost, IHostMessage, IHostMessages } from '../../models';
// noinspection TypeScriptPreferShortImport,ES6PreferShortImport
import { TextMessage } from '../TextMessage/TextMessage';

interface IProps {
   cls?: string;
   host: IHost;
   severity: HostMessageSeverity;
}

export const HostStatusMessages = React.memo(({ cls = '', severity, host }: IProps) => {
   const messages = _getMessages(host).filter(message =>
      severity === HostMessageSeverity.Error ? message.severity === HostMessageSeverity.Error : true,
   );

   return messages.length === 0 && severity === HostMessageSeverity.Info ? (
      <>{EMPTY_VALUE}</>
   ) : (
      <div className={cls}>
         {messages.map((m, i) => (
            <TextMessage key={i} title={m.source} text={m.message} isError={m.severity === HostMessageSeverity.Error} />
         ))}
      </div>
   );
});

HostStatusMessages.displayName = 'HostStatusMessages';

/**
 * Извлекает список сообщений из разных полей хоста
 *
 * Ошибкой считается только: task.error и messages (severity=error)
 * Все прочее - информационный уровень.
 *
 * Порядок:
 *    task_error
 *    messages (severity=error)
 *    messages (severity=info)
 *    task.status_message?
 *    health.message?
 *    status_reason
 *    state_reason
 */
export function _getMessages(host: IHost): IHostMessage[] {
   const result: IHostMessage[] = [];

   if (host.task && host.task.error) {
      result.push({
         message: host.task.error,
         severity: HostMessageSeverity.Error,
         source: 'Task error',
      });
   }

   if (host.messages) {
      result.push(..._extractMessagesBySeverity(host.messages, HostMessageSeverity.Error));

      result.push(..._extractMessagesBySeverity(host.messages, HostMessageSeverity.Info));
   }

   if (host.task && host.task.status_message) {
      result.push({
         message: host.task.status_message,
         severity: HostMessageSeverity.Info,
         source: 'Task status message',
      });
   }

   if (host.health && host.health.message) {
      result.push({
         message: host.health.message,
         severity: HostMessageSeverity.Info,
         source: 'Host health message',
      });
   }

   if (host.status_reason) {
      result.push({
         message: host.status_reason,
         severity: HostMessageSeverity.Info,
         source: 'Status reason',
      });
   }

   if (host.state_reason) {
      result.push({
         message: host.state_reason,
         severity: HostMessageSeverity.Info,
         source: 'State reason',
      });
   }

   result.sort((a, b) => sortHandler(_getWeight(a), _getWeight(b)));

   return result;
}

export function _extractMessagesBySeverity(messages: IHostMessages, severity: HostMessageSeverity): IHostMessage[] {
   return Object.keys(messages).reduce((acc, group) => {
      const newMessages = messages[group]
         .filter(m => m.severity === severity)
         .map(m => ({
            message: `${group}: ${m.message}`,
            severity: m.severity,
            source: 'Host messages',
         }));

      return [...acc, ...newMessages];
   }, [] as IHostMessage[]);
}

function _getWeight(m: IHostMessage) {
   return m.severity === HostMessageSeverity.Error ? 0 : 1;
}
