import { EMPTY_VALUE, IUserOrGroup, UserList, UserOrGroupType } from '@yandex-infracloud-ui/libs-next';
import * as React from 'react';
import { SyntheticEvent, useCallback, useEffect, useState } from 'react';

import { IdmRequest } from '../../models';
import { config } from '../../services';

export type Owners = Set<string> | Map<string, IdmRequest>;

interface IProps {
   cls?: string;
   editable: boolean;
   forceAvatars?: boolean;
   value: Owners;

   onChange?(e: SyntheticEvent | null, v: Owners): void;
}

/**
 * Хак, для того чтобы конвертировать магию с owners (разное расположение собачек)
 *
 * Для овнеров собака в начале - это признак группы. Без собаки - человек.
 */
const loginToOwner = (login: string): IUserOrGroup =>
   login.startsWith('@')
      ? { type: UserOrGroupType.Group, id: login.replace(/^@/, '') }
      : { type: UserOrGroupType.People, id: login };

export function outToIn(owners: Owners): IUserOrGroup[] {
   if (owners instanceof Set) {
      return Array.from(owners).map(loginToOwner);
   }

   if (owners instanceof Map) {
      return Array.from(owners.values()).map(owner => ({
         ...loginToOwner(owner.login),
         url: config.getIdmRequestOwnerUrl(owner.id),
      }));
   }

   throw new Error('unsupported value type');
}

export function intToOut(issuers: IUserOrGroup[], initialValue: Owners): Owners {
   const loginList = issuers.map(v => (v.type === UserOrGroupType.Group ? `@${v.id}` : v.id));

   if (initialValue instanceof Set) {
      return new Set(loginList);
   }

   if (initialValue instanceof Map) {
      return new Map(
         loginList.map(login => [
            login,
            initialValue.has(login) ? initialValue.get(login)! : { id: 0, login }, // маловероятно
         ]),
      );
   }

   throw new Error('unsupported value type');
}

export const OwnersList = React.memo(({ editable, onChange, value: propValue, ...props }: IProps) => {
   // hooks
   const [value, setValue] = useState<IUserOrGroup[]>([]);

   // effects
   useEffect(() => {
      setValue(outToIn(propValue));
   }, [propValue]);

   // handlers
   const update = useCallback(
      (e: SyntheticEvent | null, v: IUserOrGroup[]) => {
         if (!editable) {
            return;
         }

         setValue(v);

         if (onChange) {
            onChange(e, intToOut(v, propValue));
         }
      },
      [editable, onChange, propValue],
   );

   if (!editable && value.length === 0) return <div>{EMPTY_VALUE}</div>;

   return <UserList {...props} editable={editable} onChange={update} value={value} />;
});

OwnersList.displayName = 'OwnersList';
