import { getUniqueId, isEqual, IValueProps, Loader } from '@yandex-infracloud-ui/libs-next';
import * as React from 'react';
import { useCallback, useEffect, useMemo, useState } from 'react';

import { RestrictionOption } from './RestrictionOption';
import { IRestriction, RestrictionValue } from './RestrictionValue';

interface Props extends IValueProps<Set<string>> {
   cls?: string;
   allAvailableProjectChecks: string[];
   disabled?: boolean;
}

export const DisabledChecksEditor = React.memo(
   ({ cls = '', disabled = false, value: propValue, onChange, allAvailableProjectChecks = [] }: Props) => {
      // hooks
      const [value, setValue] = useState(new RestrictionValue(new Set(propValue), [], new Map()));
      const id = useMemo(() => getUniqueId('disabledChecksEditor'), []);

      useEffect(() => {
         if (allAvailableProjectChecks.length === 0) {
            return;
         }

         setValue(v => v.updateMapping(allAvailableProjectChecks, new Map()));
      }, [allAvailableProjectChecks]);

      useEffect(() => {
         if (!isEqual(new Set(propValue), value.valueOf())) {
            setValue(value.update(new Set(propValue)));
         }
      }, [value, propValue]);

      // handlers
      const toggle = useCallback(
         (restriction: IRestriction) => {
            const v = value.toggle(restriction.restriction);
            onChange(null, v.valueOf());
            setValue(v);
         },
         [onChange, value],
      );

      // render
      if (allAvailableProjectChecks.length === 0) {
         return <Loader cls={cls} text={'Disabled Checks loading'} align={'left'} />;
      }

      return (
         <div className={cls}>
            {value.toArray().map(r => (
               <RestrictionOption
                  key={r.restriction}
                  id={`${id}_${r.restriction}`}
                  option={r}
                  disabled={disabled}
                  onToggle={toggle}
               />
            ))}
         </div>
      );
   },
);

DisabledChecksEditor.displayName = 'DisabledChecksEditor';
