import { getUniqueId, isEqual, IValueProps, Loader } from '@yandex-infracloud-ui/libs-next';
import * as React from 'react';
import { useCallback, useEffect, useMemo, useState } from 'react';
import { useConstants } from '../reactHooks';

import { RestrictionOption } from './RestrictionOption';
import { IRestriction, RestrictionValue } from './RestrictionValue';

interface Props extends IValueProps<Set<string>> {
   cls?: string;
   disabled?: boolean;
}

export const RestrictionEditor = React.memo(({ cls = '', disabled = false, value: propValue, onChange }: Props) => {
   // hooks
   const [value, setValue] = useState(new RestrictionValue(propValue, [], new Map()));
   const id = useMemo(() => getUniqueId('restrictionEditor'), []);
   const { constants } = useConstants();

   useEffect(() => {
      if (constants === null) {
         return;
      }

      const mapping = new Map(Object.entries(constants.restrictions_dependency_mapping));

      setValue(v => v.updateMapping(constants.restrictions, mapping));
   }, [constants]);

   useEffect(() => {
      if (!isEqual(propValue, value.valueOf())) {
         setValue(value.update(propValue));
      }
   }, [value, propValue]);

   // handlers
   const toggle = useCallback(
      (restriction: IRestriction) => {
         const v = value.toggle(restriction.restriction);
         onChange(null, v.valueOf());
         setValue(v);
      },
      [onChange, value],
   );

   // render
   if (constants === null) {
      return <Loader cls={cls} text={'Restrictions loading'} align={'left'} />;
   }

   return (
      <div className={cls}>
         {value.toArray().map(r => (
            <RestrictionOption
               key={r.restriction}
               id={`${id}_${r.restriction}`}
               option={r}
               disabled={disabled}
               onToggle={toggle}
            />
         ))}
      </div>
   );
});

RestrictionEditor.displayName = 'RestrictionEditor';
