import { toggleSetItem } from '@yandex-infracloud-ui/libs-next';

export type Mapping = Map<string, string[]>;

export interface IRestriction {
   enabled: boolean;
   implicit: boolean;
   isGroup: boolean;
   restriction: string;
}

/**
 * Значение для хранения списка ограничений.
 *
 * Поведение сложное, т.к. некоторые ограничения зависят от других.
 *
 * Иммутабельный, все "мутации" возвращают новый экземпляр
 */
export class RestrictionValue {
   private readonly _value: Set<string>;

   /**
    *
    * @param value Явно включённые ограничения
    * @param _restrictions Список всех ограничений
    * @param _mapping Словарь ограничений и их зависимостей
    */
   constructor(value: Set<string>, private _restrictions: string[], private _mapping: Mapping) {
      this._value = new Set(value);
   }

   public toArray(): IRestriction[] {
      return this._restrictions.map(r => this._getFor(r));
   }

   public update(value: Set<string>): RestrictionValue {
      return new RestrictionValue(value, this._restrictions, this._mapping);
   }

   public updateMapping(restrictions: string[], mapping: Mapping): RestrictionValue {
      return new RestrictionValue(this._value, restrictions, mapping);
   }

   public toggle(restriction: string): RestrictionValue {
      const value = toggleSetItem(this._value, restriction);

      // Отключаю "дочерние" ограничения, они включены неявно
      if (this._mapping.has(restriction)) {
         this._mapping.get(restriction)!.forEach(r => value.delete(r));
      }

      return this.update(value);
   }

   public valueOf(): Set<string> {
      return new Set(this._value);
   }

   private _getFor(restriction: string): IRestriction {
      const enabled = this._value.has(restriction);

      return {
         enabled,
         implicit: this._isImplicit(restriction),
         isGroup: this._mapping.has(restriction),
         restriction,
      };
   }

   private _isImplicit(restriction: string) {
      for (const [group, subRestrictions] of Array.from(this._mapping.entries())) {
         if (this._value.has(group) && subRestrictions.includes(restriction)) {
            return true;
         }
      }

      return false;
   }
}
