import {
   autobind,
   classNames,
   IDismountedProps,
   IListOption,
   toggleSetItem,
   withDismounted,
} from '@yandex-infracloud-ui/libs';
import { TextInput } from 'lego-on-react';
import { bool, func, instanceOf, string } from 'prop-types';
import * as React from 'react';
import { KeyboardEvent, SyntheticEvent } from 'react';
import { Observable } from 'rxjs';
import { takeUntil } from 'rxjs/operators';
// noinspection TypeScriptPreferShortImport,ES6PreferShortImport
import { TagList2 } from '../TagList2/TagList2';
import styles from './TagInput.module.css';

interface IProps {
   className?: string;
   disabled?: boolean;
   id?: string;
   name?: string;
   placeholder?: string;
   value: Set<string>;

   onChange(e: SyntheticEvent | null, v: Set<string>): void;

   onBlur?(): void;

   getSuggestions?(term: string): Observable<string[]>;
}

interface IState {
   term: string;
}

class TagInput extends React.PureComponent<IProps & IDismountedProps, IState> {
   public static defaultProps = {
      disabled: false,
      id: '',
      placeholder: 'Type new tag',
   };

   public static propTypes = {
      disabled: bool,
      getSuggestions: func,
      id: string,
      name: string,
      onChange: func.isRequired,
      value: instanceOf(Set).isRequired as any, // FIXME remove as any
   };

   public state: IState = {
      term: '',
   };

   public render() {
      const items: IListOption[] = Array.from(this.props.value).map(v => ({ name: v, value: v }));

      return (
         <div className={classNames(this.props.className)}>
            {!this.props.disabled ? (
               <TextInput
                  id={this.props.id}
                  name={this.props.name}
                  theme={'normal'}
                  size={'s'}
                  placeholder={this.props.placeholder}
                  text={this.state.term}
                  onChange={this._onEdit}
                  onKeyDown={this._onKeyDown}
                  onBlur={this._tryToAdd}
                  hasClear={true}
                  suggest={Boolean(this.props.getSuggestions)}
                  showListOnFocus={true}
                  suggestUrl={''}
                  dataprovider={this._dataProvider}
                  controlAttrs={{ autoComplete: 'off' }}
               />
            ) : null}

            {items.length > 0 ? (
               <TagList2
                  className={classNames({ [styles.editable]: !this.props.disabled })}
                  items={items}
                  onRemove={this._handleRemove}
                  editable={!this.props.disabled}
               />
            ) : null}
         </div>
      );
   }

   // noinspection JSUnusedLocalSymbols
   @autobind
   private _dataProvider(url: string, text: string, cb: (result: string[]) => void): void {
      this.props.getSuggestions!(text)
         .pipe(takeUntil(this.props.dismounted!))
         .subscribe(suggestions => {
            const unused = suggestions.filter(s => !this.props.value.has(s));

            return cb(unused);
         });
   }

   @autobind
   private _handleRemove(id: string) {
      this.props.onChange(null, toggleSetItem(this.props.value, id));
   }

   @autobind
   private _onEdit(term: string): void {
      this.setState({ term });
   }

   @autobind
   private _onKeyDown(e: KeyboardEvent) {
      switch (e.key) {
         case 'Enter':
         case ',':
         case ' ':
            e.preventDefault();
            this._tryToAdd(e);
            break;

         default:
            // Пропускаем событие дальше
            break;
      }
   }

   @autobind
   private _tryToAdd(e: SyntheticEvent) {
      if (this.props.onBlur) {
         this.props.onBlur();
      }
      // Предотвращаю удаления тега при его повторном вводе
      if (this.props.value.has(this.state.term)) {
         return;
      }

      // Нечего добавлять
      if (!this.state.term.trim()) {
         return;
      }

      const value = new Set(this.props.value);
      value.add(this.state.term);

      this.props.onChange(e, value);
      this.setState({ term: '' });
   }
}

export const TagInputEnhanced = withDismounted(TagInput);
