import { Suggest } from '@yandex-data-ui/common';
import { StaffPersonOrGroup, SuggestLayer } from '@yandex-infracloud-ui/libs-next';
import { getStrategy } from '@yandex-infracloud-ui/libs';
import * as React from 'react';
import { SyntheticEvent, useCallback, useState } from 'react';

export enum UserSelectMode {
   Issuers = 'issuers',
   Owners = 'owners',
}

interface IProps {
   autofocus?: boolean;
   cls?: string;
   mode: UserSelectMode;
   name?: string;

   onSelect(e: SyntheticEvent | null, login: string | null): void;
}

const strategy = getStrategy(SuggestLayer.PeopleOrGroups);

/**
 * Нормализует значение в зависимости от режима
 *
 * owners – @ в начале означает группу, без собаки – логин
 * issuers – @ в конце означает логин со стафа, без собаки – внутренняя сущность
 *
 * экспортируется только для теста
 */
export const normalizeValue = (item: StaffPersonOrGroup | null, mode: UserSelectMode): string | null => {
   if (item === null) {
      return null;
   }

   const loginOrSlug = strategy.getQueryFromEntity(item);

   switch (mode) {
      case UserSelectMode.Owners:
         return item.layer === SuggestLayer.Groups ? `@${loginOrSlug}` : loginOrSlug;

      case UserSelectMode.Issuers:
         return item.layer === SuggestLayer.People ? `${loginOrSlug}@` : `@${loginOrSlug}`;
   }
};

export const UserSelect = React.memo(({ autofocus = false, cls = '', mode, onSelect }: IProps) => {
   const [value, setValue] = useState('');

   // handlers
   const onUpdate = useCallback(
      (item: StaffPersonOrGroup | null) => {
         onSelect(null, normalizeValue(item, mode));
      },
      [mode, onSelect],
   );

   const handleGetItems = useCallback(async (query): Promise<StaffPersonOrGroup[] | any> => {
      if (!query) return [];

      return strategy
         .load(query)
         .toPromise()
         .then(resp => resp.slice(0, 10));
   }, []);

   // render

   return (
      <Suggest
         getItems={handleGetItems}
         text={value}
         autoFocus={autofocus}
         getItemsOnMount={true}
         onUpdate={setValue}
         className={cls}
         onItemClick={onUpdate}
         applicableInputValue={true}
         placeholder={'Login or group'}
         renderItem={item => strategy.renderItem(item as any, value)}
         debounce={500}
         syncPopupOnResize={false}
      />
   );
});

UserSelect.displayName = 'UserSelect';
