import { classNames, IValueProps, mergeSets, toggleSetItem } from '@yandex-infracloud-ui/libs';
import { Button, Icon, TextInput } from 'lego-on-react';
import * as React from 'react';
import { useCallback, useEffect, useState } from 'react';
import deleteSvg from '../../actions/host_actions/components/icons/delete.svg';

import styles from './VLANsSelect.module.css';

interface IProps extends IValueProps<Set<number>> {
   items: Set<number>;
   readonly: boolean;
}

export const VLANsSelect = React.memo(({ items, readonly, value, onChange }: IProps) => {
   // hooks
   const [newItem, setNewItem] = useState('');
   const [allItems, setAllItems] = useState([] as number[]);

   // effects
   useEffect(() => {
      setAllItems(Array.from(mergeSets(items, value)));
   }, [items, value]);

   // handlers
   const updateNewItem = useCallback((v: string) => {
      setNewItem(v);
   }, []);

   const addNewItem = useCallback(() => {
      if (/^\d+$/.test(newItem)) {
         onChange(null, toggleSetItem(value, parseInt(newItem, 10)));
         setNewItem('');
      }
   }, [newItem, onChange, value]);

   const addNewItemOnEnter = useCallback(
      (e: React.KeyboardEvent<HTMLElement>) => {
         if (e.key === 'Enter') {
            addNewItem();
         }
      },
      [addNewItem],
   );

   const dataProvider = useCallback(
      (url, v, cb) => {
         cb(allItems.filter(i => !value.has(i)).map(i => i.toString()));
      },
      [allItems, value],
   );

   const remove = (item: number) => () => onChange(null, toggleSetItem(value, item));

   // render
   const deleteButtonClass = classNames(styles.delete, {
      [styles.hidden]: readonly,
   });

   return (
      <div>
         {value.size === 0 ? (
            <div className={styles.empty}>There are no VLANs</div>
         ) : (
            <div className={styles.vlanTable}>
               {Array.from(value).map(item => (
                  <div key={item} className={styles.item}>
                     <span className={styles.vlan}>{item}</span>
                     <Button
                        theme={'clear'}
                        size={'xs'}
                        cls={deleteButtonClass}
                        onClick={remove(item)}
                        disabled={readonly}
                        icon={<Icon url={deleteSvg} />}
                     />
                  </div>
               ))}
            </div>
         )}

         {readonly ? null : (
            <div className={styles.inputGroup}>
               <TextInput
                  theme={'normal'}
                  size={'s'}
                  type={'number'}
                  placeholder={'Type VLAN...'}
                  cls={styles.input}
                  name={'extra'}
                  text={newItem}
                  onChange={updateNewItem}
                  onKeyDown={addNewItemOnEnter}
                  suggest={true}
                  suggestUrl={''}
                  showListOnFocus={true}
                  dataprovider={dataProvider}
               />

               <Button theme={'normal'} size={'s'} text={'Add'} name={'add'} onClick={addNewItem} disabled={readonly} />
            </div>
         )}
      </div>
   );
});

VLANsSelect.displayName = 'VLANsSelect';
