import { HostHealthStatus, IConstantGroups, IHostHealthStatuses } from '../../models';

import { groupByType, IHostHealthGroup } from './healthGroups';

describe('healthGroups', () => {
   it('groupByType', () => {
      const health: IHostHealthStatuses = {
         group1_item1: HostHealthStatus.Passed,
         group1_item2: HostHealthStatus.Failed,
         group1_item3: HostHealthStatus.Missing,

         group2_item1: HostHealthStatus.Passed,
         group2_item2: HostHealthStatus.Passed,

         group3_item1: HostHealthStatus.Missing,
         group3_item2: HostHealthStatus.Missing,

         unexpected1: HostHealthStatus.Passed,
         unexpected2: HostHealthStatus.Passed,
      };

      const template: IConstantGroups = {
         first: ['group1_item1', 'group1_item2', 'group1_item3'],
         second: ['group2_item1', 'group2_item2'],
         third: ['group3_item1', 'group3_item2'],
      };

      const grouped = groupByType(health, template);

      expect(grouped).toEqual([
         {
            items: [
               { name: 'group1_item1', status: HostHealthStatus.Passed },
               { name: 'group1_item2', status: HostHealthStatus.Failed },
               { name: 'group1_item3', status: HostHealthStatus.Missing },
            ],
            name: 'first',
            status: HostHealthStatus.Failed,
         },
         {
            items: [
               { name: 'group2_item1', status: HostHealthStatus.Passed },
               { name: 'group2_item2', status: HostHealthStatus.Passed },
            ],
            name: 'second',
            status: HostHealthStatus.Passed,
         },
         {
            items: [
               { name: 'group3_item1', status: HostHealthStatus.Missing },
               { name: 'group3_item2', status: HostHealthStatus.Missing },
            ],
            name: 'third',
            status: HostHealthStatus.Missing,
         },
         {
            items: [
               { name: 'unexpected1', status: HostHealthStatus.Passed },
               { name: 'unexpected2', status: HostHealthStatus.Passed },
            ],
            name: 'other',
            status: HostHealthStatus.Passed,
         },
      ] as IHostHealthGroup[]);
   });

   it('groupByType partly', () => {
      const health: IHostHealthStatuses = {
         group1_item1: HostHealthStatus.Passed,
         group1_item2: HostHealthStatus.Failed,
         group1_item3: HostHealthStatus.Missing,
      };

      const template: IConstantGroups = {
         first: ['group1_item1', 'group1_item2', 'group1_item3'],
         second: ['group2_item1', 'group2_item2'],
         third: ['group3_item1', 'group3_item2'],
      };

      const grouped = groupByType(health, template);

      expect(grouped).toEqual([
         {
            items: [
               { name: 'group1_item1', status: HostHealthStatus.Passed },
               { name: 'group1_item2', status: HostHealthStatus.Failed },
               { name: 'group1_item3', status: HostHealthStatus.Missing },
            ],
            name: 'first',
            status: HostHealthStatus.Failed,
         },
      ] as IHostHealthGroup[]);
   });

   it('groupByType real (snapshot test)', () => {
      const health: IHostHealthStatuses = {
         bmc: HostHealthStatus.Passed,
         cpu: HostHealthStatus.Passed,
         cpu_capping: HostHealthStatus.Passed,
         disk: HostHealthStatus.Missing,
         fs_check: HostHealthStatus.Passed,
         link: HostHealthStatus.Passed,
         memory: HostHealthStatus.Passed,
         net64_check: HostHealthStatus.Passed,
         overheat: HostHealthStatus.Passed,
         rack: HostHealthStatus.Passed,
         reboots: HostHealthStatus.Passed,
         ssh: HostHealthStatus.Failed,
         switch: HostHealthStatus.Passed,
         tainted_kernel: HostHealthStatus.Passed,
         unreachable: HostHealthStatus.Passed,
         walle_host_certificate: HostHealthStatus.Passed,
         walle_meta: HostHealthStatus.Passed,
      };

      const template: IConstantGroups = {
         availability: ['unreachable', 'ssh'],
         hardware: ['memory', 'disk', 'link', 'bmc', 'cpu', 'overheat', 'cpu_capping'],
         infrastructure: ['switch', 'rack'],
         software: ['meta', 'walle_meta', 'reboots', 'tainted_kernel', 'fs_check'],
      };

      expect(groupByType(health, template)).toEqual([
         {
            items: [
               { name: 'unreachable', status: HostHealthStatus.Passed },
               { name: 'ssh', status: HostHealthStatus.Failed },
            ],
            name: 'availability',
            status: HostHealthStatus.Failed,
         },
         {
            items: [
               { name: 'memory', status: HostHealthStatus.Passed },
               { name: 'disk', status: HostHealthStatus.Missing },
               { name: 'link', status: HostHealthStatus.Passed },
               { name: 'bmc', status: HostHealthStatus.Passed },
               { name: 'cpu', status: HostHealthStatus.Passed },
               { name: 'overheat', status: HostHealthStatus.Passed },
               { name: 'cpu_capping', status: HostHealthStatus.Passed },
            ],
            name: 'hardware',
            status: HostHealthStatus.Missing,
         },
         {
            items: [
               { name: 'switch', status: HostHealthStatus.Passed },
               { name: 'rack', status: HostHealthStatus.Passed },
            ],
            name: 'infrastructure',
            status: HostHealthStatus.Passed,
         },
         {
            items: [
               { name: 'walle_meta', status: HostHealthStatus.Passed },
               { name: 'reboots', status: HostHealthStatus.Passed },
               { name: 'tainted_kernel', status: HostHealthStatus.Passed },
               { name: 'fs_check', status: HostHealthStatus.Passed },
            ],
            name: 'software',
            status: HostHealthStatus.Passed,
         },
         {
            items: [
               { name: 'net64_check', status: HostHealthStatus.Passed },
               { name: 'walle_host_certificate', status: HostHealthStatus.Passed },
            ],
            name: 'other',
            status: HostHealthStatus.Passed,
         },
      ] as IHostHealthGroup[]);
   });
});
