import { HostHealthStatus, IConstantGroups, IHostHealthStatuses } from '../../models';

export interface IHostHealthItem {
   name: string;
   status: HostHealthStatus;
}

export interface IHostHealthGroup {
   items: IHostHealthItem[];
   name: string;
   status: HostHealthStatus;
}

export function groupByType(health: IHostHealthStatuses, mapping: IConstantGroups): IHostHealthGroup[] {
   const allStatuses: Set<string> = Object.keys(mapping).reduce((acc, groupName) => {
      mapping[groupName].forEach(i => acc.add(i));

      return acc;
   }, new Set<string>());

   const knownItems: IHostHealthGroup[] = Object.keys(mapping)
      .map(groupName => {
         const groupItems = mapping[groupName];

         const items: IHostHealthItem[] = groupItems
            .filter(item => item in health)
            .map(item => ({
               name: item,
               status: health[item],
            }));

         return {
            items,
            name: groupName,
            status: _getItemsStatus(items),
         };
      })
      .filter(group => group.items.length > 0);

   const otherItems: IHostHealthItem[] = Object.keys(health)
      .filter(k => !allStatuses.has(k))
      .map(item => ({
         name: item,
         status: health[item],
      }));

   const result = [...knownItems];
   if (otherItems.length > 0) {
      result.push({
         items: otherItems,
         name: 'other',
         status: _getItemsStatus(otherItems),
      } as IHostHealthGroup);
   }

   return result;
}

function _getItemsStatus(items: IHostHealthItem[]): HostHealthStatus {
   // Порядок задаёт приоритет вычисления статуса группы
   // от @n-malakhov: failed -> invalid -> suspected -> staled/missing/void -> passed -> unsure
   const statuses: HostHealthStatus[] = [
      HostHealthStatus.Failed, // проверка не прошла
      HostHealthStatus.Invalid, // кривые метаданные
      HostHealthStatus.Suspected, // упало, но еще можно подождать
      HostHealthStatus.Staled, // результат проверки неактуален
      HostHealthStatus.Missing, // проверка была пропущена
      HostHealthStatus.Void, // проверка была пропущена, результаты затерялись
      HostHealthStatus.Passed, // проверка прошла успешно
      HostHealthStatus.Unsure, // результат проверки возможно неактуален
   ];

   for (const status of statuses) {
      if (items.some(i => i.status === status)) {
         return status;
      }
   }

   // fallback (для неизвестных статусов)
   return statuses[statuses.length - 1];
}
