/**
 * @file
 * В отличие от /utils/reactHooks/* тут хранятся хуки, специфичные для этого проекта
 */

import { createSelector } from '@reduxjs/toolkit';
import {
   DISMISS_REASON,
   hasIncluded,
   modalService,
   toasts,
   useBehaviourSubject,
   useBodyStyles,
   useDismounted,
   useObservableValue,
} from '@yandex-infracloud-ui/libs';
import { TextInputProps } from 'lego-on-react';
import { useCallback, useEffect, useState } from 'react';
import { useDispatch, useSelector } from 'react-redux';
import { useTitle } from 'react-use';
import { Observable, of } from 'rxjs';

import { auth, dictApi } from '../services';
import { globalState, IAutomationPlot, IConstants, IHost, IMaintenancePlot, IProject, IScenario } from '../models';
import { isActual } from '../utils/toLibs';
import { loadProjects, projectsSlice } from '../state/projects';
import { RootState } from '../state/store';
import { takeUntil } from 'rxjs/operators';

export function useCanEditProject(project: IProject): boolean {
   const user = useBehaviourSubject(globalState.user);

   return auth.canEditProject(project, user);
}

export function useCanEditHost(host: IHost): boolean {
   const user = useBehaviourSubject(globalState.user);

   return auth.canEditHost(host, user);
}

export function useCanRunHostAction(host: IHost): boolean {
   const user = useBehaviourSubject(globalState.user);

   return auth.canRunHostAction(host, user);
}

export function useCanEditPlot(plot: IAutomationPlot | IMaintenancePlot | null): boolean {
   const user = useBehaviourSubject(globalState.user);

   return auth.canEditPlot(plot, user);
}

export function useCanEditScenario(scenario: IScenario | null): boolean {
   const user = useBehaviourSubject(globalState.user);

   return auth.canEditScenario(scenario, user);
}

export function useConstants(): { constants: IConstants | null; isError: boolean } {
   const [isError, setIsError] = useState(false);

   const constants = useObservableValue(dictApi.getConstants(), () => setIsError(true), []);

   return { constants, isError };
}

export function useBodyScroll(mode: string | null) {
   useBodyStyles({ overflowY: mode || undefined });
}

/**
 * Отслеживает актуальность данных на основе временной метки. Проверяет пять раз в секунду по-умолчанию
 *
 * Оптимизирован, чтобы не приводить к ререндеру компонента без необходимости.
 *
 * @param timeout Временная метка актуальности (до этой метки значение считается актуальным).
 *        Задаётся в секундах.
 * @param interval Интервал проверки в миллисекундах. По-умолчанию 200мс, т.е. 5 раз в секунду.
 */
export function useIsActual(timeout: number, interval = 200) {
   const [stopped, setStopped] = useState(!isActual(timeout));

   useEffect(() => {
      if (stopped) {
         return;
      }

      const t = window.setInterval(() => {
         if (!isActual(timeout)) {
            setStopped(true);
            window.clearInterval(t);
         }
      }, interval);

      return () => window.clearInterval(t);
   });

   return !stopped;
}

export type SuggestionProps = Pick<TextInputProps, 'suggest' | 'suggestUrl' | 'dataprovider'>;

export function useSuggestionForLego(items: string[]): SuggestionProps {
   const dataprovider = useCallback(
      (url: string, text: string, cb: (r: string[]) => void) =>
         cb(
            items.filter(
               d =>
                  // Точное совпадение игнорим (нет смысла показывать, уже ввели)
                  d !== text && hasIncluded(text, d),
            ),
         ),
      [items],
   );

   return {
      dataprovider,
      suggest: items.length > 0,
      suggestUrl: '',
   };
}

export function useWalleTitle(title: string, suffix = 'WALL-E') {
   const value = title ? `${title} - ${suffix}` : suffix;

   useTitle(value, { restoreOnUnmount: true });
}

export function useTitleForUnselectedProjects(title: string) {
   const dispatch = useDispatch();

   useEffect(() => {
      dispatch(projectsSlice.actions.setTitleForUnselected(title));
   }, [dispatch, title]);
}

/**
 * Триггерит загрузку проектов, если они еще не былы загружены
 */
export function useProjectsLoadEnsure() {
   const dispatch = useDispatch();

   useEffect(() => {
      dispatch(loadProjects(true));
   }, [dispatch]);
}

export enum TagType {
   Deploy = 'deploy',
   Project = 'project',
}

const selectProjectsTags = createSelector(
   (s: RootState, type: TagType) => type,
   (s: RootState) => s.projects,
   (type, projects) => {
      switch (type) {
         case TagType.Project:
            return projects.projectTags;
         case TagType.Deploy:
            return projects.deployTags;
         default:
            return [];
      }
   },
);

export function useProjectsTags(type: TagType, limit = 20): (term: string) => Observable<string[]> {
   useProjectsLoadEnsure();

   const tagsSelector = useCallback((s: RootState) => selectProjectsTags(s, type), [type]);
   const tags = useSelector(tagsSelector);

   return useCallback((term: string) => of(tags.filter(tag => hasIncluded(term, tag)).slice(0, limit)), [limit, tags]);
}

export interface ModalConfig {
   off: {
      component: React.FC<any>;
      successMessage: string;
      failureMessage: string;
   };
   on: {
      component: React.FC<any>;
      successMessage: string;
      failureMessage: string;
   };
}

/**
 * Открывает модальное окно на основе переданного конфига
 *
 * @param modalConfigs
 * @param onChange
 */
export function useModals<T>(modalConfigs: Map<T, ModalConfig>, onChange: (id: string) => void) {
   const dismounted = useDismounted();
   const [submitting, setSubmitting] = useState(false);

   const openModal = useCallback(
      (id: string, type: T, targetState: boolean, props?: any) => {
         setSubmitting(true);
         const config = modalConfigs.get(type)!;
         const branch = targetState ? config.on : config.off;

         modalService
            .open(branch.component, { ...props, id }, false)
            .pipe(takeUntil(dismounted))
            .subscribe(
               () => {
                  toasts.success(branch.successMessage);
                  onChange(id);
                  setSubmitting(false);
               },
               reason => {
                  if (reason !== DISMISS_REASON) {
                     toasts.apiError(branch.failureMessage, reason);
                  }
                  setSubmitting(false);
               },
            );
      },
      [dismounted, modalConfigs, onChange],
   );

   return {
      openModal,
      submitting,
   };
}
