import { DEFAULT_MAX_BUSY_HOSTS, IAutomationLimits, IConstants, IdmRequest, IProject } from '../../models';
import { IProjectReports } from '../../rich_shared';
import { ThunkAction } from 'redux-thunk';
import { catchError, takeUntil } from 'rxjs/operators';
import { forkJoin, of, Subject } from 'rxjs';
import { config, projectApi } from '../../services';
import { projectsSlice } from '../projects';
import { toasts } from '@yandex-infracloud-ui/libs';
import {
   AsyncLoadedProjectOwners,
   AutomationsFormParams,
   BasicFormParams,
   PlotsFormParams,
   CmsFormParams,
   DeployingFormParams,
   FullProjectFormState,
   NotificationsFormParams,
   OthersFormParams,
   ProfilingFormParams,
   VlansFormParams,
   extractAutomationFormParams,
   CmsType,
   initialState,
   CAuthFormParams,
} from './models';

export enum ActionType {
   AddOwner = 'FullProjectForm:AddOwner',
   Clear = 'FullProjectForm:Clear',
   SetBasicFields = 'FullProjectForm:SetBasicFields',
   SetCAuthFields = 'FullProjectForm:SetCAuthFields',
   SetCms = 'FullProjectForm:SetCms',
   AddCms = 'FullProjectForm:AddCms',
   SetVlans = 'FullProjectForm:SetVlans',
   SetDeploying = 'FullProjectForm:SetDeploying',
   SetProfiling = 'FullProjectForm:SetProfiling',
   SetAutomationLimits = 'FullProjectForm:SetAutomationLimits',
   SetHostLimits = 'FullProjectForm:SetHostLimits',
   SetPlots = 'FullProjectForm:SetPlots',
   SetRestrictions = 'FullProjectForm:SetRestrictions',
   SetDisabledChecks = 'FullProjectForm:SetDisabledChecks',
   SetNotifications = 'FullProjectForm:SetNotifications',
   SetReports = 'FullProjectForm:SetReports',
   SetAutomations = 'FullProjectForm:SetAutomations',
   SetOthers = 'FullProjectForm:SetOthers',
   SetNewProject = 'FullProjectForm:SetNewProject',
   SetProject = 'FullProjectForm:SetProject',
   Reset = 'FullProjectForm:Reset',
}

//region Sync actions
export const addOwner = (login: string) => ({
   login,
   type: ActionType.AddOwner as ActionType.AddOwner,
});

export const clear = () => ({
   type: ActionType.Clear as ActionType.Clear,
});

export const setBasicFields = (params: BasicFormParams) => ({
   params,
   type: ActionType.SetBasicFields as ActionType.SetBasicFields,
});

export const setCAuthFields = (params: CAuthFormParams) => ({
   params,
   type: ActionType.SetCAuthFields as ActionType.SetCAuthFields,
});

export const setCms = (
   cmsSettings: typeof initialState.cmsSettings,
   params: CmsFormParams | null,
   index: number | null,
   constants: IConstants,
) => ({
   cmsSettings,
   params,
   index,
   constants,
   type: ActionType.SetCms as ActionType.SetCms,
});

export const addCms = (constants: IConstants) => ({
   constants,
   params: {
      _isTvmRequired: true,
      _type: CmsType.Default,
      max_busy_hosts: DEFAULT_MAX_BUSY_HOSTS,
      url: 'default',
   },
   type: ActionType.AddCms as ActionType.AddCms,
});

export const setVlans = (params: VlansFormParams) => ({
   params,
   type: ActionType.SetVlans as ActionType.SetVlans,
});

export const setDeploying = (params: DeployingFormParams, constants: IConstants) => ({
   constants,
   params,
   type: ActionType.SetDeploying as ActionType.SetDeploying,
});

export const setProfiling = (params: ProfilingFormParams) => ({
   params,
   type: ActionType.SetProfiling as ActionType.SetProfiling,
});

export const setAutomationLimits = (params: IAutomationLimits) => ({
   params,
   type: ActionType.SetAutomationLimits as ActionType.SetAutomationLimits,
});

export const setHostLimits = (params: IAutomationLimits) => ({
   params,
   type: ActionType.SetHostLimits as ActionType.SetHostLimits,
});

export const setPlots = (params: PlotsFormParams) => ({
   params,
   type: ActionType.SetPlots as ActionType.SetPlots,
});

export const setRestrictions = (params: Set<string>) => ({
   params,
   type: ActionType.SetRestrictions as ActionType.SetRestrictions,
});

export const setDisabledChecks = (params: Set<string>) => ({
   params,
   type: ActionType.SetDisabledChecks as ActionType.SetDisabledChecks,
});

export const setNotifications = (params: NotificationsFormParams) => ({
   params,
   type: ActionType.SetNotifications as ActionType.SetNotifications,
});

export const setReports = (params: IProjectReports) => ({
   params,
   type: ActionType.SetReports as ActionType.SetReports,
});

export const setAutomations = (params: AutomationsFormParams) => ({
   params,
   type: ActionType.SetAutomations as ActionType.SetAutomations,
});

export const setOthers = (params: OthersFormParams) => ({
   params,
   type: ActionType.SetOthers as ActionType.SetOthers,
});

export const setProject = (params: IProject, owners: AsyncLoadedProjectOwners) => ({
   owners,
   params,
   type: ActionType.SetProject as ActionType.SetProject,
});

export const setNewProject = () => ({
   type: ActionType.SetNewProject as ActionType.SetNewProject,
});

export const reset = () => ({
   type: ActionType.Reset as ActionType.Reset,
});

export type SyncAction =
   | ReturnType<typeof addOwner>
   | ReturnType<typeof clear>
   | ReturnType<typeof setBasicFields>
   | ReturnType<typeof setCAuthFields>
   | ReturnType<typeof setCms>
   | ReturnType<typeof addCms>
   | ReturnType<typeof setDisabledChecks>
   | ReturnType<typeof setVlans>
   | ReturnType<typeof setDeploying>
   | ReturnType<typeof setProfiling>
   | ReturnType<typeof setAutomationLimits>
   | ReturnType<typeof setHostLimits>
   | ReturnType<typeof setPlots>
   | ReturnType<typeof setRestrictions>
   | ReturnType<typeof setNotifications>
   | ReturnType<typeof setReports>
   | ReturnType<typeof setAutomations>
   | ReturnType<typeof setOthers>
   | ReturnType<typeof setProject>
   | ReturnType<typeof setNewProject>
   | ReturnType<typeof reset>;
//endregion

//region Async actions
type ThunkResult<R> = ThunkAction<R, { fullProjectForm: FullProjectFormState }, undefined, SyncAction>;

export const loadProject = (projectId: string, dismounted: Subject<unknown>): ThunkResult<void> => dispatch => {
   const project$ = projectApi.getById(projectId);
   let requested$ = of(new Map<string, IdmRequest>());
   let revoking$ = of(new Map<string, IdmRequest>());

   if (!config.isExternal) {
      requested$ = projectApi.getRequestedOwners(projectId).pipe(
         // Неудачный запрос владельцев не должен ломать страницу
         catchError(err => {
            toasts.apiError('Requested owners loading from IDM', err);

            return of(new Map<string, IdmRequest>()); // default value
         }),
      );

      revoking$ = projectApi.getRevokingOwners(projectId).pipe(
         // Неудачный запрос владельцев не должен ломать страницу
         catchError(err => {
            toasts.apiError('Revoking owners loading from IDM', err);

            return of(new Map<string, IdmRequest>()); // default value
         }),
      );
   }

   forkJoin({
      project: project$,
      requestedOwners: requested$,
      revokingOwners: revoking$,
   })
      .pipe(takeUntil(dismounted))
      .subscribe(
         ({ project, requestedOwners, revokingOwners }) =>
            dispatch(setProject(project, { requestedOwners, revokingOwners })),
         toasts.handleApiError('Project loading'),
      );
};

export const loadAutomations = (projectId: string, dismounted: Subject<unknown>): ThunkResult<void> => dispatch => {
   projectApi
      .getById(projectId, ['dns_automation', 'dns_domain', 'healing_automation', 'fsm_handbrake'])
      .pipe(takeUntil(dismounted))
      .subscribe(project => {
         const params = extractAutomationFormParams(project);
         dispatch(setAutomations(params));

         // sync changes in another state branch (/projects)
         dispatch(projectsSlice.actions.setAutomations({ projectId, params }) as any);
      }, toasts.handleApiError('Project loading'));
};

type AsyncAction = ReturnType<typeof loadProject> | ReturnType<typeof loadAutomations>;
//endregion

export type Action = SyncAction | AsyncAction;
