import { getValidResult, IValidationResult } from '@yandex-infracloud-ui/libs';
import { IProjectReports } from '../../rich_shared';
import {
   CmsValue,
   DEFAULT_CERTIFICATE_DEPLOY,
   DEFAULT_MAX_BUSY_HOSTS,
   DefaultDeployNetwork,
   DefaultProvisioner,
   FsmHandbrakePayload,
   Automation,
   IAutomationLimits,
   ICustomCms,
   IDefaultCms,
   IProject,
   NotificationRecipients,
   ProjectOwners,
   VlanSchema,
   createEmptyNotificationRecipients,
   DefaultDeployConfigPolicy,
   createProjectOwners,
   CAuthFlowType,
   ICAuthSettings,
} from '../../models';

export enum CmsType {
   Default = 'default',
   Custom = 'custom',
}

export interface CmsFormParams {
   _isTvmRequired?: boolean;
   _type: CmsType;
   api_version?: string;
   max_busy_hosts?: number;
   tvm_app_id?: number;
   url: 'default' | string;
   temporary_unreachable_enabled?: boolean;
}

export type AsyncLoadedProjectOwners = Pick<ProjectOwners, 'requestedOwners' | 'revokingOwners'>;

export type BasicFormParams = Pick<
   IProject,
   'name' | 'id' | 'tags' | 'bot_project_id' | 'owners' | 'roles' | 'yc_iam_folder_id' | 'type'
> & {
   _owners?: ProjectOwners;
};

export type VlansFormParams = Pick<
   IProject,
   'owned_vlans' | 'vlan_scheme' | 'hbf_project_id' | 'ip_method' | 'native_vlan' | 'extra_vlans' | 'use_fastbone'
>;

export type DeployingFormParams = Pick<
   IProject,
   | '_deployConfig'
   | 'deploy_config'
   | 'deploy_config_policy'
   | 'provisioner'
   | 'deploy_network'
   | 'deploy_tags'
   | 'certificate_deploy'
>;

export type ProfilingFormParams = Pick<IProject, 'profile' | 'profile_tags'>;

export type PlotsFormParams = Pick<
   IProject,
   'automation_limits' | 'automation_plot_id' | 'host_limits' | 'maintenance_plot_id'
>;

export type CAuthFormParams = ICAuthSettings;

export interface NotificationsFormParams {
   recipients: NotificationRecipients;
}

export interface AutomationsFormParams {
   dns: boolean;
   dns_automation: Automation;
   dns_domain: string | undefined;
   fsm: boolean;
   fsmDetails?: FsmHandbrakePayload;
   healing: boolean;
   healing_automation: Automation;
}

export interface DnsZoneFormParams {
   domain?: string;
   link?: string;
   dnsZoneID?: string;
}

export type OthersFormParams = Pick<IProject, 'reboot_via_ssh' | 'tier'>;

export interface FormParams<T> {
   params: T;
   validation: IValidationResult<T>;
}

export function getFormParams<T>(params: T): FormParams<T> {
   return {
      params,
      validation: getValidResult<T>(),
   };
}

function createEmptyReports() {
   return {
      enabled: false,
      extra: {},
      queue: '',
      summary: '',
   };
}

export const initialState = {
   allAvailableProjectChecks: getFormParams<string[]>([]),
   automationLimits: getFormParams<IAutomationLimits>({}),

   automations: getFormParams<AutomationsFormParams>({
      dns: false,
      dns_domain: '',
      dns_automation: {
         enabled: false,
      },
      fsm: false,
      healing: false,
      healing_automation: {
         enabled: false,
      },
   }),

   basic: getFormParams<BasicFormParams>({
      _owners: createProjectOwners({}),
      bot_project_id: undefined,
      id: '',
      name: '',
      owners: undefined,
      tags: new Set(),
      roles: {},
      yc_iam_folder_id: undefined,
      type: undefined,
   }),

   cauth: getFormParams<CAuthFormParams>({
      flow_type: CAuthFlowType.Classic,
      insecure_ca_list_url: '',
      key_sources: [],
      krl_url: '',
      secure_ca_list_url: '',
      sudo_ca_list_url: '',
      trusted_sources: [],
   }),

   plots: getFormParams<PlotsFormParams>({
      automation_limits: {},
      automation_plot_id: undefined,
      maintenance_plot_id: undefined,
   }),

   cmsSettings: [
      getFormParams<CmsFormParams>({
         _isTvmRequired: true,
         _type: CmsType.Default,
         max_busy_hosts: DEFAULT_MAX_BUSY_HOSTS,
         url: 'default',
      }),
   ],

   dnsZone: getFormParams<DnsZoneFormParams>({
      domain: '',
      link: '',
   }),

   deploying: getFormParams<DeployingFormParams>({
      _deployConfig: {
         config: '',
         provisioner: DefaultProvisioner,
      },
      deploy_config_policy: DefaultDeployConfigPolicy,
      certificate_deploy: false,
      deploy_config: '',
      deploy_network: DefaultDeployNetwork,
      deploy_tags: new Set(),
      provisioner: DefaultProvisioner,
   }),

   isIdChangedManually: false,

   isSaving: false,

   isNewProject: null as boolean | null,

   hostLimits: getFormParams<IAutomationLimits>({}),

   notifications: getFormParams<NotificationsFormParams>({
      recipients: createEmptyNotificationRecipients(),
   }),

   disabledChecks: getFormParams<Set<string>>(new Set()),

   reports: getFormParams<IProjectReports>(createEmptyReports()),

   originalProject: null as any | null,

   profiling: getFormParams<ProfilingFormParams>({
      profile: '',
      profile_tags: new Set(),
   }),

   restrictions: getFormParams<Set<string>>(new Set()),

   vlans: getFormParams<VlansFormParams>({
      // TODO уточнить дефолты
      extra_vlans: new Set(),
      hbf_project_id: '',
      ip_method: 'hostname',
      native_vlan: undefined,
      owned_vlans: new Set(),
      vlan_scheme: VlanSchema.None,
      use_fastbone: true,
   }),

   others: getFormParams<OthersFormParams>({
      reboot_via_ssh: false,
   }),
};

export type FullProjectFormState = typeof initialState;

function cmsToFormParams(cms: CmsValue): CmsFormParams {
   if (!cms) {
      return {
         _isTvmRequired: true,
         _type: CmsType.Default,
         max_busy_hosts: DEFAULT_MAX_BUSY_HOSTS,
         url: 'default',
      };
   }

   if (cms.url === 'default') {
      const defaultCms = cms as IDefaultCms;

      return {
         _isTvmRequired: true,
         _type: CmsType.Default,
         max_busy_hosts: defaultCms.max_busy_hosts,
         url: defaultCms.url,
      };
   }

   const customCms = cms as ICustomCms;

   return {
      _isTvmRequired: true,
      _type: CmsType.Custom,
      api_version: customCms.api_version,
      tvm_app_id: customCms.tvm_app_id,
      temporary_unreachable_enabled: customCms.temporary_unreachable_enabled,
      url: customCms.url,
   };
}

export function formParamsToCms(params: CmsFormParams): CmsValue {
   switch (params._type) {
      case CmsType.Custom: {
         return {
            url: params.url,
            api_version: params.api_version,
            tvm_app_id: params.tvm_app_id,
            temporary_unreachable_enabled: params.temporary_unreachable_enabled,
         } as ICustomCms;
      }

      case CmsType.Default: {
         return {
            max_busy_hosts: params.max_busy_hosts,
            url: 'default',
         } as IDefaultCms;
      }
   }
}

export const extractAutomationFormParams = (project: IProject): AutomationsFormParams => ({
   // DNS
   dns: project.dns_automation!.enabled,
   dns_domain: project.dns_domain || '',
   dns_automation: project.dns_automation ?? initialState.automations.params.dns_automation,

   // FSM
   fsm: Boolean(project.fsm_handbrake),
   fsmDetails: project.fsm_handbrake,

   // Healing
   healing: project.healing_automation!.enabled,
   healing_automation: project.healing_automation ?? initialState.automations.params.healing_automation,
});

export const projectToFormParams = (
   state: FullProjectFormState,
   project: IProject,
   owners: AsyncLoadedProjectOwners,
): FullProjectFormState => ({
   ...state,
   allAvailableProjectChecks: getFormParams<string[]>(project.all_available_project_checks || []),
   automationLimits: getFormParams(project.automation_limits!),
   automations: getFormParams(extractAutomationFormParams(project)),
   cauth: getFormParams<CAuthFormParams>({ ...project.cauth_settings }),
   basic: getFormParams<BasicFormParams>({
      _owners: createProjectOwners({ ...owners, actualOwners: project.owners }),
      bot_project_id: project.bot_project_id,
      id: project.id,
      name: project.name,
      owners: project.owners,
      tags: project.tags,
      type: project.type,
      roles: project.roles,
      yc_iam_folder_id: project.yc_iam_folder_id,
   }),
   plots: getFormParams<PlotsFormParams>({
      automation_limits: project._checks,
      automation_plot_id: project.automation_plot_id,
      maintenance_plot_id: project.maintenance_plot_id,
   }),
   cmsSettings: project.cms!.map(e => getFormParams(cmsToFormParams(e as CmsValue))),
   deploying: getFormParams<DeployingFormParams>({
      _deployConfig: {
         config: project.deploy_config!,
         provisioner: project.provisioner!,
      },
      deploy_config_policy: project.deploy_config_policy,
      certificate_deploy: project.certificate_deploy || DEFAULT_CERTIFICATE_DEPLOY,
      deploy_config: project.deploy_config,
      deploy_network: project.deploy_network || DefaultDeployNetwork,
      deploy_tags: project.deploy_tags,
      provisioner: project.provisioner,
   }),
   disabledChecks: getFormParams<Set<string>>(project.manually_disabled_checks || new Set()),
   dnsZone: getFormParams<DnsZoneFormParams>({
      domain: project?.yc_dns_zone?.domain,
      link: project?.yc_dns_zone?.link,
      dnsZoneID: project.yc_dns_zone_id,
   }),
   isIdChangedManually: true,
   isNewProject: false,
   isSaving: false,
   hostLimits: getFormParams(project.host_limits!),
   notifications: getFormParams<NotificationsFormParams>({
      recipients: project.notifications ? project.notifications.recipients : createEmptyNotificationRecipients(),
   }),
   reports: getFormParams<IProjectReports>(project.reports ? project.reports : createEmptyReports()),
   others: getFormParams<OthersFormParams>({
      reboot_via_ssh: project.reboot_via_ssh,
      tier: project.tier,
   }),
   profiling: getFormParams<ProfilingFormParams>({
      profile: project.profile,
      profile_tags: project.profile_tags,
   }),
   restrictions: getFormParams<Set<string>>(project.default_host_restrictions || new Set()),
   vlans: getFormParams<VlansFormParams>({
      extra_vlans: project.extra_vlans,
      hbf_project_id: project.hbf_project_id,
      ip_method: project.ip_method,
      native_vlan: project.native_vlan,
      owned_vlans: project.owned_vlans,
      vlan_scheme: project.vlan_scheme,

      // дефолтное значение. не приходит с апи
      use_fastbone: true,
   }),
});

export const isProjectFormValid = (v: FullProjectFormState): boolean => {
   const subForms: (keyof FullProjectFormState)[] = [
      'automationLimits',
      'automations',
      'basic',
      'plots',
      'deploying',
      'notifications',
      'profiling',
      'restrictions',
      'vlans',
   ];

   return subForms.every((f: any) =>
      Array.isArray(v[f]) ? v[f].every((e: any) => e.validation.isValid) : v[f].validation.isValid,
   );
};
