import { getValidResult, IValidationResult } from '@yandex-infracloud-ui/libs';
import { config } from 'services';

import { DefaultDeployNetwork, DefaultProvisioner, IConstants, Provisioner } from '../../models';
import { createError, createFieldValidator } from '../../utils/forTests';
import { BasicFormParams, CmsType, DeployingFormParams, ProfilingFormParams } from './models';
import {
   basicValidatorExternal,
   basicValidatorInternal,
   cmsValidator,
   deployingValidator,
   profilingValidator,
} from './validators';

const deployTagPattern = '[a-zA-Z0-9_=:-]';
const projectTagPattern = '[a-zA-Z0-9_.:-]';

interface TestTagsConfig<F extends keyof T, T> {
   checkCount?: boolean;
   context?: any;
   fieldName: F;
   isFieldValid: (f: F, v: T[F] | any, context?: any) => IValidationResult<T>;
   pattern: string;
}

// Вынес дублирующийся код проверки тегов (теги проекта, deploy_tags, profile_tags проверяются одинаково)
function testTags<T, F extends keyof T>({
   isFieldValid,
   fieldName,
   pattern,
   checkCount = true,
   context,
}: TestTagsConfig<F, T>): void {
   it('should be valid', () => {
      expect(isFieldValid(fieldName, new Set(), context)).toEqual(getValidResult());
      expect(isFieldValid(fieldName, new Set(['xxx', 'yyy']), context)).toEqual(getValidResult());
   });

   it('should be invalid if contains invalid characters', () => {
      expect(isFieldValid(fieldName, new Set(['?&&&:']), context)).toEqual(
         createError(fieldName as string, [`"?&&&:" is invalid tag. Tag must contain only ${pattern} symbols.`]),
      );
   });

   if (checkCount) {
      it('should be invalid if more then 20', () => {
         const items = [];
         for (let i = 0; i < 30; i += 1) {
            items.push(`tag-${i.toString()}`);
         }
         expect(isFieldValid(fieldName, new Set(items), context)).toEqual(
            createError(fieldName as string, [`${fieldName} field must have less than or equal to 20 items`]),
         );
      });
   }
}

describe('state|fullProjectForm/validators', () => {
   describe('BasicFormParams', () => {
      const isFieldValid = createFieldValidator(
         {
            bot_project_id: 1,
            id: 'test',
            name: 'test',
            owners: new Set(),
            tags: new Set(),
            yc_iam_folder_id: undefined,
         } as BasicFormParams,
         config.isExternal ? basicValidatorExternal : basicValidatorInternal,
      );

      describe('should validate bot_project_id', () => {
         it('should be valid', () => {
            expect(isFieldValid('bot_project_id', 454545454)).toEqual(getValidResult());
         });

         it('should be invalid', () => {
            expect(isFieldValid('bot_project_id', null as any)).toEqual(
               createError('bot_project_id', ['BOT Project ID must be an integer']),
            );

            expect(isFieldValid('bot_project_id', -1)).toEqual(
               createError('bot_project_id', ['BOT Project ID must be a positive integer']),
            );

            expect(isFieldValid('bot_project_id', 2.5)).toEqual(
               createError('bot_project_id', ['BOT Project ID must be a positive integer']),
            );
         });
      });

      describe('should validate id', () => {
         it('should be valid', () => {
            expect(isFieldValid('id', 'xxx')).toEqual(getValidResult());
            expect(isFieldValid('id', 'xxx-yyy')).toEqual(getValidResult());
            expect(isFieldValid('id', 'x0123')).toEqual(getValidResult());
            expect(isFieldValid('id', 'x0123456789-x0123456789')).toEqual(getValidResult());
         });

         it('should be invalid', () => {
            expect(isFieldValid('id', 'x')).toEqual(
               createError('id', ['Project ID/slug must be at least 2 characters']),
            );

            expect(isFieldValid('id', 'x0123456789-x0123456789-x0123456789-x0123456789')).toEqual(
               createError('id', ['Project ID/slug must be at most 32 characters']),
            );

            // noinspection LongLine
            const error = `Project ID/slug must have value like "qloud-common-dev-mtn", "yt-arnold-vla-masters". Regexp is /^[a-z][a-z\\d]*(?:-[a-z\\d]+)*$/`;

            expect(isFieldValid('id', 'xxx.yyy')).toEqual(createError('id', [error]));

            expect(isFieldValid('id', '0123')).toEqual(createError('id', [error]));

            expect(isFieldValid('id', 'XXXX')).toEqual(createError('id', [error]));
         });
      });

      describe('should validate name', () => {
         it('should be valid', () => {
            expect(isFieldValid('name', 'Any short name')).toEqual(getValidResult());
         });

         it('should be invalid', () => {
            expect(isFieldValid('name', '')).toEqual(createError('name', ['Project name is a required field']));

            expect(isFieldValid('name', 'x0123456789-x0123456789-x0123456789-x0123456789')).toEqual(
               createError('name', ['Project name must be at most 32 characters']),
            );
         });
      });

      describe('should validate owners', () => {
         it('should be valid', () => {
            expect(isFieldValid('owners', new Set())).toEqual(getValidResult());
            expect(isFieldValid('owners', new Set(['khoden']))).toEqual(getValidResult());
            expect(isFieldValid('owners', new Set(['@khoden']))).toEqual(getValidResult());
         });

         it('should be invalid', () => {
            const error = createError('owners', [
               'Project owner must be nickname or service, e.g. "login" or "@service"',
            ]);

            expect(isFieldValid('owners', new Set(['khoden@yandex-team.ru']))).toEqual(error);

            // noinspection LongLine
            expect(
               isFieldValid(
                  'owners',
                  new Set(['khoden123456789khoden123456789khoden123456789khoden123456789khoden123456789']),
               ),
            ).toEqual(error);

            // noinspection LongLine
            expect(
               isFieldValid(
                  'owners',
                  new Set([
                     '@khoden123456789khoden123456789khoden123456789khoden123456789khoden123456789khoden123456789khoden123456789khoden123456789khoden123456789khoden123456789khoden123456789khoden123456789khoden123456789khoden123456789khoden123456789',
                  ]),
               ),
            ).toEqual(error);
         });
      });

      describe('should validate tags', () => {
         testTags({ isFieldValid, fieldName: 'tags', pattern: projectTagPattern, checkCount: false });
      });
   });

   describe('CmsFormParams', () => {
      const constants = {
         cms_api_versions: ['v1.3', 'v1.2', 'v1.1', 'v1.0'],
      } as IConstants;
      const context = { constants };

      describe('default CMS', () => {
         it('should be valid', () => {
            expect(cmsValidator({ _type: CmsType.Default, url: 'default', max_busy_hosts: 1 }, context)).toEqual(
               getValidResult(),
            );
         });

         it('should be valid with max_busy_hosts greater then 1', () => {
            expect(cmsValidator({ _type: CmsType.Default, url: 'default', max_busy_hosts: 422 }, context)).toEqual(
               getValidResult(),
            );
         });

         it('should be invalid without max_busy_hosts', () => {
            expect(cmsValidator({ _type: CmsType.Default, url: 'default', api_version: 'v1.3' }, context)).toEqual(
               createError('max_busy_hosts', ['max_busy_hosts is a required field']),
            );
         });

         it('should be invalid if max_busy_hosts less then 1', () => {
            expect(cmsValidator({ _type: CmsType.Default, url: 'default', max_busy_hosts: 0 }, context)).toEqual(
               createError('max_busy_hosts', ['max_busy_hosts must be greater than or equal to 1']),
            );
         });

         it('should be invalid if url is not default', () => {
            expect(cmsValidator({ _type: CmsType.Default, url: 'xxx', max_busy_hosts: 1 }, context)).toEqual(
               createError('url', ['url must be valid "default"']),
            );
         });
      });

      describe('custom CMS', () => {
         const validCmsUrl = 'http://clusterstate.yandex-team.ru/api/v1';

         it('should be valid with valid version', () => {
            expect(
               cmsValidator(
                  {
                     _type: CmsType.Custom,
                     api_version: 'v1.0',
                     tvm_app_id: 222,
                     url: validCmsUrl,
                  },
                  context,
               ),
            ).toEqual(getValidResult());
         });

         it('should be invalid without tvm_app_id in not YP version', () => {
            expect(cmsValidator({ _type: CmsType.Custom, api_version: 'v1.3', url: validCmsUrl }, context)).toEqual(
               createError('tvm_app_id', ['TVM App ID is a required field']),
            );
         });

         it('should be invalid with invalid url', () => {
            expect(
               cmsValidator(
                  {
                     _type: CmsType.Custom,
                     api_version: 'v1.3',
                     tvm_app_id: 222,
                     url: 'xxsdfyyy',
                  },
                  context,
               ),
            ).toEqual(createError('url', ['url must be valid url']));
         });

         it('should be invalid with wrong api_version', () => {
            expect(
               cmsValidator({ _type: CmsType.Custom, url: validCmsUrl, tvm_app_id: 222, api_version: 'x' }, context),
            ).toEqual(createError('api_version', ['api_version has unsupported CMS version']));
         });
      });
   });

   describe('DeployFormParams', () => {
      const constants = {
         provisioners: [Provisioner.EINE, Provisioner.LUI],
      } as IConstants;
      const context = { constants };

      const isFieldValid = createFieldValidator(
         {
            _deployConfig: { provisioner: DefaultProvisioner, config: 'test-ubuntu' },
            certificate_deploy: false,
            deploy_config: 'test-ubuntu',
            deploy_network: DefaultDeployNetwork,
            provisioner: DefaultProvisioner,
         } as DeployingFormParams,
         deployingValidator,
      );

      describe('should validate deploy_tags', () => {
         testTags({ isFieldValid, fieldName: 'deploy_tags', pattern: deployTagPattern, context });
      });

      describe('should validate provisioner', () => {
         it('should be valid', () => {
            expect(isFieldValid('provisioner', Provisioner.LUI, context)).toEqual(getValidResult());
            expect(isFieldValid('provisioner', Provisioner.EINE, context)).toEqual(getValidResult());
         });

         it('should be invalid', () => {
            expect(isFieldValid('provisioner', '', context)).toEqual(
               createError('provisioner', ['Provisioner is a required field']),
            );

            expect(isFieldValid('provisioner', 'xxx', context)).toEqual(
               createError('provisioner', ['Provisioner must be one of supported provisioners']),
            );
         });
      });
   });

   describe('ProfilingFormParams', () => {
      const isFieldValid = createFieldValidator(
         {
            profile: 'common-ubuntu',
            profile_tags: new Set(),
         } as ProfilingFormParams,
         profilingValidator,
      );

      describe('should validate profile', () => {
         it('should be valid', () => {
            expect(isFieldValid('profile', 'image-name')).toEqual(getValidResult());
         });
      });

      describe('should validate profile_tags', () => {
         testTags({ isFieldValid, fieldName: 'profile_tags', pattern: deployTagPattern });
      });
   });
});
