/* eslint-disable no-template-curly-in-string */
import { createValidator, setAsArray } from '@yandex-infracloud-ui/libs';
import { boolean, number, object, Schema, string, array } from 'yup';
import { CAuthFormParams } from '.';

import {
   DeployNetwork,
   deployTagSchema,
   IConstants,
   IDeployConfig,
   OWNER_REGEXP,
   PROJECT_ID_REGEXP,
   projectTagSchema,
   Provisioner,
   URL_REGEXP,
   CAuthFlowType,
   CAuthKeySources,
   CAuthTrustedSources,
} from '../../models';

import { BasicFormParams, CmsFormParams, CmsType, DeployingFormParams, ProfilingFormParams } from './models';

function checkCmsVersion(version: string) {
   // @ts-ignore
   const constants: IConstants | null = this.options.context.constants;
   if (!constants) {
      throw new Error('Need constants for validation');
   }

   return constants ? constants.cms_api_versions.includes(version) : false;
}

function checkProvisioner(value: Provisioner) {
   // @ts-ignore
   const constants: IConstants = this.options.context.constants;
   if (!constants) {
      throw new Error('Need constants for validation');
   }

   return constants.provisioners.includes(value);
}

// noinspection LongLine
const basicShape = {
   bot_project_id: number()
      .label('BOT Project ID')
      .required('${path} is a required field')
      .typeError('${path} must be an integer')
      .integer('${path} must be a positive integer')
      .positive('${path} must be a positive integer'),

   // TODO проверить на уникальность
   id: string()
      .label('Project ID/slug')
      .required()
      .min(2)
      .max(32)
      .matches(
         PROJECT_ID_REGEXP,
         `$\{path} must have value like "qloud-common-dev-mtn", "yt-arnold-vla-masters". Regexp is ${PROJECT_ID_REGEXP}`,
      ),

   // TODO проверить на уникальность
   name: string().label('Project name').required().min(1).max(32),

   tags: projectTagSchema,
};
export const basicValidatorExternal = createValidator<BasicFormParams>({
   ...basicShape,
   yc_iam_folder_id: string().required('Folder ID is a required field'),
});
export const basicValidatorInternal = createValidator<BasicFormParams>({
   ...basicShape,
   owners: setAsArray().of(
      string().matches(OWNER_REGEXP, 'Project owner must be nickname or service, e.g. "login" or "@service"'),
   ) as any,
});

export const cauthValidator = createValidator<CAuthFormParams>({
   flow_type: string().oneOf(Object.values(CAuthFlowType)) as any,

   insecure_ca_list_url: string().matches(URL_REGEXP, {
      message: 'url must be valid',
      excludeEmptyString: true,
   }),

   key_sources: array().of(string().oneOf(Object.values(CAuthKeySources))) as any,

   krl_url: string().matches(URL_REGEXP, {
      message: 'url must be valid',
      excludeEmptyString: true,
   }),

   secure_ca_list_url: string().matches(URL_REGEXP, {
      message: 'url must be valid',
      excludeEmptyString: true,
   }),

   sudo_ca_list_url: string().matches(URL_REGEXP, {
      message: 'url must be valid',
      excludeEmptyString: true,
   }),

   trusted_sources: array().when('flow_type', {
      is: CAuthFlowType.Classic,
      otherwise: array(),
      then: array().of(string().oneOf(Object.values(CAuthTrustedSources))) as any,
   }) as any,
});

export const cmsValidator = createValidator<CmsFormParams>({
   _isTvmRequired: boolean(),
   _type: string().required().oneOf([CmsType.Default, CmsType.Custom]) as any, // FIXME typings

   api_version: string().when('_type', {
      is: 'default',
      otherwise: string().required().test('cmsVersion', '${path} has unsupported CMS version', checkCmsVersion),
      then: string().notRequired(),
   }),

   max_busy_hosts: number().when('_type', {
      is: 'default',
      otherwise: number().notRequired(),
      then: number().required().min(1),
   }),

   /*
    TVM-app применим только для custom cms, для default и YP эта настройка неприменима.
    TVM-app является обязательным полем при изменении настроек cms.
    */
   tvm_app_id: number().when(['_type', '_isTvmRequired'], {
      is: (_type: CmsType, _isTvmRequired: boolean = true) => _type === CmsType.Custom && _isTvmRequired,
      otherwise: number().label('TVM App ID').positive().notRequired(),
      then: number().label('TVM App ID').positive().required(),
   }),

   url: string()
      .when('_type', {
         is: 'default',
         then: string().required().oneOf(['default'], '${path} must be valid "default"'),
      })
      .when(['_type', 'api_version'], {
         is: (_type: CmsType) => _type === CmsType.Custom,
         then: string().required().matches(URL_REGEXP, '${path} must be valid url'),
      }),
});

const provisionerSchema = string()
   .label('Provisioner')
   .required()
   .test('provisioner', '${path} must be one of supported provisioners', checkProvisioner) as Schema<Provisioner>;

export const deployingValidator = createValidator<DeployingFormParams>({
   _deployConfig: object().shape<IDeployConfig>({
      config: string().label('Deploy config').required(),
      provisioner: provisionerSchema,
   }),
   certificate_deploy: boolean().required(),
   deploy_config: string().required(),
   deploy_network: string().when('provisioner', {
      is: Provisioner.LUI,
      otherwise: string(),
      then: string()
         .required()
         .oneOf([DeployNetwork.Project, DeployNetwork.Service] as string[]),
   }) as Schema<DeployNetwork>,
   deploy_tags: deployTagSchema,
   provisioner: provisionerSchema,
});

export const profilingValidator = createValidator<ProfilingFormParams>({
   profile: string(), // TODO проерять вхождение в список существующих
   profile_tags: deployTagSchema,
});
