import { toasts } from '@yandex-infracloud-ui/libs';
import { ThunkAction } from 'redux-thunk';
import { Subject } from 'rxjs';
import { map } from 'rxjs/operators';

import { globalState, IUserInfo, UserModel } from '../../models';
import { dictApi } from '../../services';
import { yandexCookies } from '../../services/yandexCookies';
// noinspection TypeScriptPreferShortImport
import { projectsSlice } from '../projects/slice';

import { globalsSlice, GlobalsState } from './slice';

type ThunkResult<R> = ThunkAction<R, { globals: GlobalsState }, undefined, any>;

export const loadConstants = (dismounted: Subject<unknown>, skipIfLoaded = true): ThunkResult<void> => (
   dispatch,
   getState,
) => {
   if (skipIfLoaded && getState().globals.constants) {
      return;
   }

   dictApi
      .getConstants()
      .subscribe(
         constants => dispatch(globalsSlice.actions.setConstants(constants)),
         toasts.handleApiError('Constants loading'),
      );
};

export const loadUser = (skipIfLoaded = true): ThunkResult<void> => (dispatch, getState) => {
   if (skipIfLoaded && getState().globals.user) {
      return;
   }

   const userId = yandexCookies.getYandexUID();

   dictApi
      .getCurrentUser(!skipIfLoaded)
      .pipe(
         map(
            (apiUser: IUserInfo) =>
               new UserModel(
                  userId,
                  apiUser.login,
                  apiUser.login,
                  apiUser.admin,
                  'api',
                  apiUser.groups,
                  apiUser.projects,
                  apiUser.project_roles,
                  apiUser.automation_plots,
               ),
         ),
      )
      .subscribe(user => {
         dispatch(globalsSlice.actions.setUser(user));
         dispatch(projectsSlice.actions.setUserProjects(user.projects));
         globalState.user.next(user); // deprecated
      }, toasts.handleApiError('Current user loading'));
};
