import Fuse, { FuseOptions } from 'fuse.js';
import { OwnershipValue } from '../../models';
import { listToEntityStore } from '../commonModels';
import { AutomationFilterValue, ProjectFilters, ProjectItem } from './models';
import { filterItems, fuzzySearchProjects, isAllSelected } from './slice';

describe('state|projects/slice', () => {
   describe('filterItems', () => {
      const items: ProjectItem[] = [
         {
            dns_automation: { enabled: true },
            healing_automation: { enabled: true },
            id: 'a',
            name: 'Alpha',
            tags: ['rtc'],
         },
         {
            dns_automation: { enabled: false },
            healing_automation: { enabled: true },
            id: 'b',
            name: 'Beta',
         },
         {
            dns_automation: { enabled: false },
            healing_automation: { enabled: false },
            id: 'c',
            name: 'Gamma',
            tags: ['rtc', 'mtn'],
         },
         {
            dns_automation: { enabled: true },
            healing_automation: { enabled: false },
            id: 'd',
            name: 'Delta',
         },
      ];
      const store = listToEntityStore(items);

      describe('filter by ownership', () => {
         it(`should filter by ownership=${OwnershipValue.All}`, () => {
            const filters: ProjectFilters = {
               automation: AutomationFilterValue.All,
               ownership: OwnershipValue.All,
               query: '',
            };

            expect(filterItems(store, filters, ['a', 'd'])).toEqual(['a', 'b', 'c', 'd']);
         });

         it(`should filter by ownership=${OwnershipValue.My}`, () => {
            const filters: ProjectFilters = {
               automation: AutomationFilterValue.All,
               ownership: OwnershipValue.My,
               query: '',
            };

            expect(filterItems(store, filters, ['a', 'd'])).toEqual(['a', 'd']);
         });
      });

      describe('filter by automation', () => {
         it(`should filter by automation=${AutomationFilterValue.All}`, () => {
            const filters: ProjectFilters = {
               automation: AutomationFilterValue.All,
               ownership: OwnershipValue.All,
               query: '',
            };

            expect(filterItems(store, filters, ['a', 'd'])).toEqual(['a', 'b', 'c', 'd']);
         });

         it(`should filter by automation=${AutomationFilterValue.AllOff}`, () => {
            const filters: ProjectFilters = {
               automation: AutomationFilterValue.AllOff,
               ownership: OwnershipValue.All,
               query: '',
            };

            expect(filterItems(store, filters, ['a', 'd'])).toEqual(['c']);
         });

         it(`should filter by automation=${AutomationFilterValue.AllOn}`, () => {
            const filters: ProjectFilters = {
               automation: AutomationFilterValue.AllOn,
               ownership: OwnershipValue.All,
               query: '',
            };

            expect(filterItems(store, filters, ['a', 'd'])).toEqual(['a']);
         });

         it(`should filter by automation=${AutomationFilterValue.DnsOff}`, () => {
            const filters: ProjectFilters = {
               automation: AutomationFilterValue.DnsOff,
               ownership: OwnershipValue.All,
               query: '',
            };

            expect(filterItems(store, filters, ['a', 'd'])).toEqual(['b', 'c']);
         });

         it(`should filter by automation=${AutomationFilterValue.HealingOff}`, () => {
            const filters: ProjectFilters = {
               automation: AutomationFilterValue.HealingOff,
               ownership: OwnershipValue.All,
               query: '',
            };

            expect(filterItems(store, filters, ['a', 'd'])).toEqual(['c', 'd']);
         });
      });

      describe('filter by query', () => {
         it('should filter by id', () => {
            const filters: ProjectFilters = {
               automation: AutomationFilterValue.All,
               ownership: OwnershipValue.All,
               query: 'b',
            };

            expect(filterItems(store, filters, [])).toEqual(['b']);
         });

         it('should filter by name', () => {
            const filters: ProjectFilters = {
               automation: AutomationFilterValue.All,
               ownership: OwnershipValue.All,
               query: 'alpha',
            };

            expect(filterItems(store, filters, [])).toEqual(['a']);
         });

         it('should filter by tag', () => {
            const filters: ProjectFilters = {
               automation: AutomationFilterValue.All,
               ownership: OwnershipValue.All,
               query: '#rtc',
            };

            expect(filterItems(store, filters, [])).toEqual(['a', 'c']);
         });

         it('should filter by tags', () => {
            const filters: ProjectFilters = {
               automation: AutomationFilterValue.All,
               ownership: OwnershipValue.All,
               query: '#rtc #mtn',
            };

            expect(filterItems(store, filters, [])).toEqual(['c']);
         });

         it('should filter by complex query', () => {
            const filters: ProjectFilters = {
               automation: AutomationFilterValue.All,
               ownership: OwnershipValue.All,
               query: 'alp #rtc',
            };

            expect(filterItems(store, filters, [])).toEqual(['a']);
         });

         it('should filter with empty tag', () => {
            const filters: ProjectFilters = {
               automation: AutomationFilterValue.All,
               ownership: OwnershipValue.All,
               query: 'alp #',
            };

            expect(filterItems(store, filters, [])).toEqual(['a']);
         });
      });

      describe('fuzzy search', () => {
         xit('find proper options', () => {
            const options: FuseOptions<ProjectItem> = {
               includeScore: true,
               keys: ['id', 'name', 'tags'],
               shouldSort: true,
            };
            const fuse = new Fuse(items, options);

            const query = 'mtn rtc';
            const result = fuse.search(query);

            console.log(query);
            console.table(result.map((r: any) => ({ ...r.item, score: r.score })));

            expect(result).toBeTruthy();
         });

         it('should return list of IDs', () => {
            expect(fuzzySearchProjects(store, ['a', 'b', 'c', 'd'], '#rtc')).toEqual(['a', 'c']);
         });
      });
   });

   describe('isAllSelected', () => {
      it('should be false for empty', () => {
         expect(isAllSelected([], {})).toBe(false);
         expect(isAllSelected([], { a: true })).toBe(false);
      });

      it('should be true for equal', () => {
         expect(isAllSelected(['a', 'b'], { b: true, a: true })).toBe(true);
      });

      it('should be false for not equal', () => {
         expect(isAllSelected(['a', 'b'], { b: true })).toBe(false);
      });
   });
});
