import { createSlice, PayloadAction } from '@reduxjs/toolkit';
import { formatNumber, isEmpty, isEqual, sortHandler } from '@yandex-infracloud-ui/libs';
import Fuse from 'fuse.js';

import { OwnershipValue } from '../../models';
import { config } from '../../services';
import { parseHostQuery } from '../../services/api/api_helpers';

import { EntityStore, IdSet, idSetToList, listToEntityStore, listToIdSet, toggleListItem } from '../commonModels';
import { AutomationsFormParams } from '../fullProjectForm';
import {
   AutomationFilterValue,
   initialState,
   ProjectFilters,
   ProjectItem,
   ProjectState,
   TypeFilterValueAll,
} from './models';

interface SetAutomationPayload {
   params: AutomationsFormParams;
   projectId: string;
}

const FUSE_OPTIONS: Fuse.IFuseOptions<ProjectItem> = {
   keys: [
      {
         name: 'id',
         weight: 0.7,
      },
      {
         name: 'name',
         weight: 0.3,
      },
   ],
   shouldSort: true,
};

export function fuzzySearchProjects(store: EntityStore<ProjectItem>, ids: string[], query: string): string[] {
   if (query.length === 0) {
      return ids;
   }

   const { names, tags } = parseHostQuery(query);

   const items = ids
      .map(id => store.byIds[id])
      .filter(project => {
         if (tags.length === 0) {
            return true;
         }

         // Если есть теги, предварительно фильтруем входные данные
         return project.tags && project.tags.length > 0 ? tags.every(t => project.tags!.includes(t)) : false;
      });

   // Запрос пустой (кроме тегов), возвращаем как есть
   if (names.length === 0) {
      return items.map(p => p.id);
   }

   const fuse = new Fuse(items, FUSE_OPTIONS);

   return fuse.search(names.join(' ')).map(i => i.item.id);
}

export function filterItems(
   store: EntityStore<ProjectItem>,
   filters: ProjectFilters,
   userProjects: string[],
): string[] {
   const userProjectsSet = new Set(userProjects);

   const obviousFiltered = store.allIds.filter(id => {
      const item = store.byIds[id];
      // ownership
      const byOwnership = filters.ownership === OwnershipValue.My ? userProjectsSet.has(item.id) : true;

      // automation
      let byAutomation: boolean;
      switch (filters.automation) {
         case AutomationFilterValue.All:
            byAutomation = true;
            break;

         case AutomationFilterValue.AllOff:
            byAutomation = Boolean(
               item.dns_automation &&
                  !item.dns_automation.enabled &&
                  item.healing_automation &&
                  !item.healing_automation.enabled,
            );
            break;

         case AutomationFilterValue.AllOn:
            byAutomation = Boolean(
               item.dns_automation &&
                  item.dns_automation.enabled &&
                  item.healing_automation &&
                  item.healing_automation.enabled,
            );
            break;

         case AutomationFilterValue.DnsOff:
            byAutomation = Boolean(item.dns_automation && !item.dns_automation.enabled);
            break;

         case AutomationFilterValue.HealingOff:
            byAutomation = Boolean(item.healing_automation && !item.healing_automation.enabled);
            break;

         case AutomationFilterValue.FsmOn:
            byAutomation = Boolean(item.fsm_handbrake && item.fsm_handbrake.timeout_time * 1000 > Date.now());
            break;

         default:
            byAutomation = true;
            break;
      }

      // type
      const byType = filters.type !== TypeFilterValueAll.All ? filters.type === item?.type : true;

      return byOwnership && byAutomation && byType;
   });

   const q = filters.query.trim().toLowerCase();

   return fuzzySearchProjects(store, obviousFiltered, q);
}

function _getTagsByPopularity(projects: ProjectItem[], getTags: (p: ProjectItem) => string[] | undefined): string[] {
   const tagsCounted = new Map<string, number>();

   // Считаю популярность тегов
   projects
      .map(getTags)
      .filter(tags => !isEmpty(tags))
      .forEach(tags => {
         tags!.forEach(tag => {
            const count = (tagsCounted.get(tag) || 0) + 1;
            tagsCounted.set(tag, count);
         });
      });

   // Сортирую по популярности (популярные сверху)
   const sorted = Array.from(tagsCounted.entries());
   sorted.sort((a, b) => sortHandler(b[1], a[1]));

   return sorted.map(i => i[0]);
}

export function isAllSelected(filteredIds: string[], selectedIds: IdSet): boolean {
   return !isEmpty(filteredIds) && isEqual(listToIdSet(filteredIds), selectedIds);
}

function applyFilters(state: ProjectState): void {
   state.filteredIds = filterItems(state.allItems, state.filters, state.userProjects);
}

function updateTitle(state: ProjectState): void {
   const selected = idSetToList(state.selectedIds);

   switch (selected.length) {
      case 0:
         state.title = state.titleForUnselected;
         break;

      case 1:
         const project = state.allItems.byIds[selected[0]];
         state.title = project ? project.name : selected[0];
         break;

      default:
         state.title = `Selected projects: ${formatNumber(selected.length)}`;
         break;
   }
}

function updateTags(state: ProjectState): void {
   const projectItems = Object.values(state.allItems.byIds);

   state.deployTags = _getTagsByPopularity(projectItems, p => p.deploy_tags);
   state.projectTags = _getTagsByPopularity(projectItems, p => p.tags);
}

function select(state: ProjectState, items: string[]) {
   state.selectedIds = listToIdSet(items);
   state.isAllSelected = isAllSelected(state.filteredIds, state.selectedIds);

   updateTitle(state);
}

export const projectsSlice = createSlice({
   initialState,
   name: 'projects',
   reducers: {
      addProject(state, action: PayloadAction<ProjectItem>) {
         state.allItems.allIds.push(action.payload.id);
         state.allItems.byIds[action.payload.id] = action.payload;

         applyFilters(state);
         updateTags(state);
      },

      removeProject(state, action: PayloadAction<string>) {
         state.allItems.allIds = state.allItems.allIds.filter(id => id !== action.payload);
         delete state.allItems.byIds[action.payload];

         applyFilters(state);
         // updateTags(state);
      },

      select(state, action: PayloadAction<string[]>) {
         select(state, action.payload);
      },

      setAutomations(state, action: PayloadAction<SetAutomationPayload>) {
         const project = state.allItems.byIds[action.payload.projectId];
         if (!project) {
            return;
         }

         project.dns_automation = { enabled: action.payload.params.dns };
         project.fsm_handbrake = action.payload.params.fsmDetails;
         project.healing_automation = { enabled: action.payload.params.healing };

         applyFilters(state);
      },

      setFilters(state, action: PayloadAction<ProjectFilters>) {
         state.filters = action.payload;

         applyFilters(state);
      },

      setLoaded(state, action: PayloadAction<ProjectItem[]>) {
         state.allItems = listToEntityStore(action.payload);
         state.isLoading = false;
         state.wasLoaded = true;
         state.requested = false;

         applyFilters(state);
         updateTitle(state);
         updateTags(state);
      },

      setLoading(state, action: PayloadAction<boolean>) {
         state.isLoading = action.payload;
      },

      setRequested(state) {
         state.requested = true;
      },

      setOwnershipFilter(state, action: PayloadAction<OwnershipValue>) {
         if (state.filters.ownership !== action.payload) {
            state.filters.ownership = action.payload;
         }
      },

      setTitleForUnselected(state, action: PayloadAction<string>) {
         state.titleForUnselected = action.payload;

         updateTitle(state);
      },

      setUserProjects(state, action: PayloadAction<string[]>) {
         state.userProjects = action.payload;

         applyFilters(state);
      },

      toggleAllItems(state) {
         if (isEmpty(state.filteredIds)) {
            return;
         }

         const isSelectedAll = isAllSelected(state.filteredIds, state.selectedIds);
         select(state, isSelectedAll ? [] : state.filteredIds);
      },

      toggleItem(state, action: PayloadAction<string>) {
         const selected = toggleListItem(Object.keys(state.selectedIds), action.payload);

         select(state, selected);
      },

      toggleSidebar(state) {
         state.isCollapsed = !state.isCollapsed;

         config.isProjectListCollapsed = state.isCollapsed; // FIXME грязно!
      },
   },
});
