import { autobind, IDismountedProps, json, Loader, toasts, withDismounted } from '@yandex-infracloud-ui/libs';
import * as React from 'react';
import { Subscription } from 'rxjs';
import { finalize, takeUntil } from 'rxjs/operators';

import { ILocation, IUserInfo } from '../models';
import { getDefaultHostFilters } from '../rich_shared/host_list';
import { dictApi, hostApi } from '../services';

import { toastsSupportDecorator } from './decorators/toastsSupportDecorator';
import { IStoryDefaults } from './typings';

interface IState {
   constants?: any;
   isLoading?: boolean;
   locations?: ILocation[];
   user?: IUserInfo;
}

class ApiStories extends React.PureComponent<IDismountedProps, IState> {
   public static defaultProps = {};

   public static propTypes = {};

   public state: IState = {
      isLoading: false,
   };

   private _subscription: Subscription | null = null;

   public render() {
      return (
         <>
            <button type={'button'} onClick={this._getConstants}>
               getConstants
            </button>
            <button type={'button'} onClick={this._getLocation}>
               getLocation
            </button>
            <button type={'button'} onClick={this._getOnlyLastRequest}>
               getOnlyLastRequest (click quickly)
            </button>
            <button type={'button'} onClick={this._getUser}>
               getUser
            </button>
            <button type={'button'} onClick={this._resetState}>
               resetState
            </button>

            <Loader visible={this.state.isLoading} inline={true} />

            <pre>{json(this.state)}</pre>
         </>
      );
   }

   @autobind
   private _getConstants() {
      this.setState({ isLoading: true }, () => {
         dictApi
            .getConstants()
            .pipe(
               finalize(() => this.setState({ isLoading: false })),
               takeUntil(this.props.dismounted!),
            )
            .subscribe(constants => this.setState({ constants }), toasts.handleApiError('Loading constants'));
      });
   }

   @autobind
   private _getLocation() {
      this.setState({ isLoading: true }, () => {
         dictApi
            .getLocations()
            .pipe(
               finalize(() => this.setState({ isLoading: false })),
               takeUntil(this.props.dismounted!),
            )
            .subscribe(resp => this.setState({ locations: resp.result }), toasts.handleApiError('Loading locations'));
      });
   }

   @autobind
   private _getOnlyLastRequest() {
      if (this._subscription) {
         this._subscription.unsubscribe();
      }

      this.setState({ isLoading: true }, () => {
         this._subscription = hostApi
            .getList({
               filters: getDefaultHostFilters(),
               limit: 205,
               nextCursor: 0,
               useOwnershipFilter: false,
            })
            .pipe(
               finalize(() => {
                  this._subscription = null;
                  this.setState({ isLoading: false });
               }),
               takeUntil(this.props.dismounted!),
            )
            .subscribe(resp => console.log(resp), toasts.handleApiError('Loading hosts'));
      });
   }

   @autobind
   private _getUser() {
      this.setState({ isLoading: true }, () => {
         dictApi
            .getCurrentUser()
            .pipe(
               finalize(() => this.setState({ isLoading: false })),
               takeUntil(this.props.dismounted!),
            )
            .subscribe(user => this.setState({ user }), toasts.handleApiError('Loading user'));
      });
   }

   @autobind
   private _resetState() {
      this.setState({
         constants: undefined,
         isLoading: false,
         locations: undefined,
         user: undefined,
      });
   }
}

const ApiStoriesEnhanced = withDismounted(ApiStories);

export const regularStory = () => <ApiStoriesEnhanced />;

export default {
   decorators: [toastsSupportDecorator],
   title: 'services|API service',
} as IStoryDefaults;
