import { IValidationResult, round, Validator } from '@yandex-infracloud-ui/libs';

export function createIdGenerator() {
   let lastId = 0;

   return () => {
      lastId += 1;

      return lastId;
   };
}

export function measureDuration<R>(fn: () => R): [number, R] {
   const start = Date.now();

   const result = fn();

   return [Date.now() - start, result];
}

export function getIntRandom(from: number, to: number): number {
   const length = Math.abs(to - from); // На всякий случай

   return round(Math.random() * length + from);
}

export function createError(field: string, errors: string[]): IValidationResult<any> {
   return {
      errors: new Map([[field, errors]]),
      isValid: false,
   };
}

function serializeValidationResult<T>(r: IValidationResult<T>): string {
   const fields = Array.from(r.errors.entries()).map(([k, v]) => `${k}(${v.join(',')})`);

   return `${r.isValid ? 'VALID' : 'INVALID'}:${fields.join('; ')}`;
}

export function addValidationMatchers() {
   const success = { message: () => '', pass: true };

   expect.extend({
      toBeInvalid(actual: IValidationResult<any>, field: string, errors: string[]) {
         if (actual.isValid) {
            return {
               message: () => 'expected to be invalid',
               pass: false,
            };
         }

         const expected = serializeValidationResult(createError(field, errors));
         const actualSerialized = serializeValidationResult(actual);

         if (expected !== actualSerialized) {
            return {
               message: () => `expected to be "${expected}", but received "${actualSerialized}"`,
               pass: false,
            };
         }

         return success;
      },
      toBeValid(actual: IValidationResult<any>) {
         if (actual.isValid) {
            return success;
         }

         return {
            message: () => 'expected to be valid',
            pass: false,
         };
      },
   });
}

export function createFieldValidator<T, F extends keyof T>(
   validValue: T,
   validator: Validator<T>,
): (f: F, v: T[F], context?: any) => IValidationResult<T> {
   return (f: F, v: T[F], context = {}) => {
      const params: T = { ...validValue };
      params[f] = v;

      return validator(params, context);
   };
}

export type StubValues<T> = { [k in keyof T]: T[k][] };

/**
 * Генерирует случайный объект из заданный значений.
 *
 * Удобен, когда нужно сравнить два алгоритма на идентичность,
 * сгенерировав им на вход случайны объект тысячи раз.
 *
 * @param values Возможные значения для всех полей объекта.
 * @param count Максимальное число вариантов
 */
export function* createStubGenerator<T>(values: StubValues<T>, count: number): IterableIterator<T> {
   while (count > 0) {
      count -= 1;
      yield Object.keys(values).reduce((acc, field) => {
         const variants = values[field];
         acc[field] = variants[getIntRandom(0, variants.length - 1)];

         return acc;
      }, {} as T);
   }
}
