import abc
import datetime

import yt.yson as yson
from yp.client import YpClient as BaseYpClient
from yp_proto.yp.client.api.proto import object_service_pb2


def yson_loads(data, default=None):
    """

    :type data: six.text_type
    :type default: any
    :rtype: any
    """
    res = yson.loads(data)
    if isinstance(res, yson.YsonEntity):
        return default
    return res


class YpObject(object):
    __metaclass__ = abc.ABCMeta
    _attrs_last_modifications = None

    def __init__(self, data, mtime=()):
        """

        :type data: list[str]
        :type mtime: list[int]
        """
        self._attrs_last_modifications = {}
        for i, (attr_name, parser) in enumerate(self.selectors.values()):
            setattr(self, attr_name, parser(data[i]))
            if mtime and mtime[i]:
                m_datetime = datetime.datetime.utcfromtimestamp(int(mtime[i] / 2 ** 30))
                self._attrs_last_modifications[attr_name] = m_datetime

    @abc.abstractproperty
    def object_type(self):
        """

        :rtype: int
        """
        pass

    @abc.abstractproperty
    def selectors(self):
        """

        :rtype: collections.OrderedDict[six.text_type, tuple[six.text_type, collections.Callable]]
        """

    def get_attr_last_modification_dt(self, attr_name):
        """

        :type attr_name: six.text_type
        :rtype: datetime.datetime | None
        """
        return self._attrs_last_modifications.get(attr_name, None)


class YpClient(object):
    @classmethod
    def get_clients_by_clusters(cls, yp_config):
        """

        :type yp_config: dict
        :rtype: dict[six.text_type, YpClient]
        """
        clients = {}
        yp_discovery = yp_config.get('enable_master_discovery', True)
        yp_cfg = {
            'token': yp_config.get('token'),
            'enable_master_discovery': yp_discovery
        }
        for cluster_config in yp_config.get('clusters'):
            address = cluster_config.get('address')
            yp_cluster_name = cluster_config.get('name')
            client_base = BaseYpClient(address=address, config=yp_cfg)
            stub = client_base.create_grpc_object_stub()
            clients[yp_cluster_name] = YpClient(yp_cluster_name, stub)

        return clients

    def __init__(self, yp_cluster_name, stub):
        """

        :type yp_cluster_name: six.text_type
        :type stub: any
        """
        self.stub = stub
        self.yp_cluster_name = yp_cluster_name

    def _select_objects_values(self, obj_cls, limit, query=None, continuation_token=None):
        """

        :type obj_cls: YpObject
        :type limit: int
        :type continuation_token: six.text_type | None
        :type query: six.text_type | None
        :rtype: tuple[list[tuple[six.test_type]], six.text_type]
        """
        req = object_service_pb2.TReqSelectObjects()
        req.object_type = obj_cls.object_type
        req.limit.value = limit
        req.selector.paths.extend(obj_cls.selectors)
        if query:
            req.filter.query = query
        if continuation_token:
            req.options.continuation_token = continuation_token

        req.options.fetch_timestamps = True
        req.format = object_service_pb2.EPayloadFormat.Value("PF_YSON")
        resp = self.stub.SelectObjects(req)
        objects = [obj_cls([v.yson for v in r.value_payloads], r.timestamps) for r in resp.results]
        return objects, resp.continuation_token

    def list_objects(self, obj_cls, query='true', batch_size=200):
        """

        :type obj_cls: YpObject
        :type query: six.text_type
        :type batch_size: int
        :rtype:
        """
        continuation_token = None
        while True:
            objs_values, continuation_token = self._select_objects_values(
                obj_cls=obj_cls,
                limit=batch_size,
                query=query,
                continuation_token=continuation_token
            )
            if not objs_values:
                return
            for obj in objs_values:
                yield obj

    def get_object(self, obj_cls, obj_id, ignore_nonexistent=False):
        """

        :type obj_cls: YpObject
        :type obj_id: six.text_type
        :type ignore_nonexistent: bool
        :rtype: object | None
        """
        req = object_service_pb2.TReqGetObject()
        req.object_type = obj_cls.object_type
        req.object_id = obj_id
        req.selector.paths.extend(obj_cls.selectors)
        req.options.ignore_nonexistent = ignore_nonexistent
        response = self.stub.GetObject(req)
        if not response.HasField('result'):
            return None
        return obj_cls(response.result.values)

    def set_object_label(self, obj_cls, obj_id, label_name, label_value):
        """

        :type obj_cls: YpObject
        :type obj_id: six.text_type
        :type label_name: six.text_type
        :type label_value: six.text_type
        :rtype: object_service_pb2.TRspUpdateObject
        """
        req = object_service_pb2.TReqUpdateObject()
        req.object_type = obj_cls.object_type
        req.object_id = obj_id

        upd = req.set_updates.add()
        upd.path = label_name
        upd.value = yson.dumps(label_value)
        return self.stub.UpdateObject(req)
