import argparse
import errno
import logging
import os
import signal
import sys

from infra.ya_salt import cmd
from infra.ya_salt.lib import logutil
from infra.ya_salt.lib import config
from infra.ya_salt.lib import constants

log = logging.getLogger('ya-salt')


def main(argv):
    logutil.basic_config()
    config_pb = config.defaults()
    conf_parser = argparse.ArgumentParser(add_help=False)
    # Global params
    conf_parser.add_argument('--hostctl', help='Hostctl binary path (default: %(default)s)', default=config_pb.hostctl_path)
    conf_parser.add_argument('--report-addrs',
                             help='Override autodiscovered hm-report addresses (host1:port1,host2:port2,...)',
                             default=None)
    conf_parser.add_argument('--version', action='store_true', help='Print version and exit')
    conf_args = conf_parser.parse_known_args(args=argv)[0]
    if conf_args.version:
        from library.python import svn_version
        print 'revision:', svn_version.svn_revision()
        return True
    # Loading configuration
    parser = argparse.ArgumentParser(parents=[conf_parser])
    parser.add_argument('--debug', action="store_true", help='Debug logging', default=False)
    parser.add_argument('--log-file', type=str, help='Write log to file', default=config_pb.log_file)
    parser.add_argument('--console', action='store_true', help='Duplicate log to stderr', default=False)
    parser.add_argument('--nocolor', action='store_true', help='Disable colorize', default=False)
    subparsers = parser.add_subparsers(help='List command for ya-salt')
    # Add commands
    for mod in cmd.all_commands:
        mod.add_command(subparsers)
    args = parser.parse_args()
    config_pb.hostctl_path = args.hostctl
    if args.report_addrs:
        config_pb.report_addrs.extend(args.report_addrs.split(','))
    if args.handle in (cmd.run.handler, cmd.grains.handler):
        return args.handle(args, config_pb)
    else:
        return args.handle(args)


def setup_process():
    # Kill ourselves if running for too long
    signal.alarm(constants.SIGALARM_SEC)
    # Force timezone to avoid checking /etc/localtime on every log string
    os.putenv('TZ', 'Europe/Moscow')


if __name__ == "__main__":
    setup_process()
    try:
        ok = main(sys.argv[1:])
    except EnvironmentError as e:
        # It is okay if we get closed pipe.
        if e.errno != errno.EPIPE:
            log.exception('Unexpected error.')
            ok = False
        else:
            ok = True
    except KeyboardInterrupt as e:
        log.info('=== Interrupted, quitting... ===')
        ok = True
    except SystemExit as e:
        ok = False
        # See below for motivation
        os._exit(e.code)
    except Exception as e:
        log.exception('Unexpected error.')
        ok = False
    # Speed up exit
    try:
        sys.stdout.flush()
        sys.stderr.flush()
    except EnvironmentError as e:
        if e.errno == errno.EPIPE:
            pass
        raise
    # No need to run GC, releasing memory, etc...
    # Saves noticeable amount of time in interactive sessions (tens of millis at least)
    os._exit(0 if ok else 1)
    raise SystemExit(0 if ok else 1)
