"""
CMD ya-salt override
"""
from __future__ import print_function

import errno
import logging
import os
import shutil
import subprocess

from infra.ya_salt.lib import constants

log = logging.getLogger('override')

DIRS = [
    'components',
    'deploy',
]


def add_command(action):
    p = action.add_parser('override', help='Copy component to override dir')
    p.add_argument('-r', '--revert', help='revert override for component',
                   action='store_true')
    p.add_argument('-n', '--name', help='component name, e.g netconfig or dns.', required=True)
    p.set_defaults(handle=handler)


def revert(name):
    """
    Reverts component override by removing component directories.
    Also removes "common" directory if not overrides left after that.
    """
    for d in DIRS:
        common = os.path.join(constants.REPO_OVERRIDE_DIR, 'common', d)
        p = os.path.join(common, name)
        if os.path.exists(p):
            try:
                shutil.rmtree(p)
            except EnvironmentError as e:
                if e.errno == errno.ENOENT:
                    continue
        try:
            empty = len(os.listdir(common)) == 0
        except EnvironmentError as e:
            if e.errno == errno.ENOENT:
                continue
            print("Failed to check if '{}' is empty: {}".format(common, e))
            return False
        if empty:
            try:
                os.rmdir(common)
            except EnvironmentError as e:
                print("Failed to rmdir('{}'): {}".format(common, e))
                return False
    common = os.path.join(constants.REPO_OVERRIDE_DIR, 'common')
    if os.path.exists(common) and not os.listdir(common):
        try:
            os.rmdir(common)
        except EnvironmentError as e:
            print("Failed to rmdir('{}'): {}".format(common, e))
            return False
    subprocess.call(['tree', '-L', '3', constants.REPO_OVERRIDE_DIR])
    return True


def override(name):
    for d in DIRS:
        src = os.path.join(constants.LOCAL_REPO_CURRENT, 'common', d, name)
        if not os.path.isdir(src):
            print("No '{}/{}' directory in local repo at '{}'. Maybe a misprint?".format(
                d, name, constants.LOCAL_REPO_CURRENT
            ))
            return False
        dst = os.path.join(constants.REPO_OVERRIDE_DIR, 'common', d, name)
        if os.path.exists(dst):
            shutil.rmtree(dst)
        print("Copying '{}' -> '{}'...".format(src, dst))
        try:
            shutil.copytree(src, dst)
        except Exception as e:
            print('Copy failed: {}'.format(e))
            return False
    # For better UX, move to override directory and print it's contents.
    print("Done! Now you can edit files in '{}' "
          'to override your component!'.format(constants.REPO_OVERRIDE_DIR))
    print('Please! Be careful and cleanup after all. Thank you.')
    print('-' * 80)
    subprocess.call(['tree', '-L', '3', constants.REPO_OVERRIDE_DIR])
    return True


def handler(args):
    if args.revert:
        return revert(args.name)
    else:
        return override(args.name)
