"""
CMD ya-salt run
"""
import logging

import random
import sys
import time

from library.python import svn_version

from infra.ya_salt.hostmanager import shim

from infra.ya_salt.lib import constants
from infra.ya_salt.lib import context
from infra.ya_salt.lib import initial_setup
from infra.ya_salt.lib import logutil
from infra.ya_salt.lib import output
from infra.ya_salt.lib import policy
from infra.ya_salt.lib import runutil

log = logging.getLogger('ya-salt')


def add_command(action):
    p = action.add_parser('run', help='Run ya-salt')
    p.add_argument('--random-sleep',
                   type=int, help='Random sleep before start')
    p.add_argument('--no-orly', action="store_true", help='Disable ORLY for this operation',
                   default=False)
    p.add_argument('--local',
                   help='Use local mode for salt - provide dir with repo contents',)
    p.add_argument('--fail-fast', action="store_true",
                   help='Exit immediately after first failure')
    p.set_defaults(handle=handler)


def handler(args, conf_pb):
    log_level = logging.INFO
    if args.debug or conf_pb.debug_log:
        log_level = logging.DEBUG
    manual = sys.stdin.isatty()
    # Force console output if we're logging to terminal
    logutil.setup_logging(log_level, args.log_file, manual)
    if args.no_orly:
        conf_pb.no_orly = True
    ok, _ = runutil.lock_run(constants.F_LOCK_FR)
    if not ok:
        # Do not log as fatal if run from cron - it is okay
        if manual:
            l = log.fatal
        else:
            l = log.info
        l("{} is locked - exiting.".format(constants.F_LOCK_FR))
        return True
    log.info("=== ya-salt=2.0-{} started as '{}' ===".format(
        svn_version.svn_revision(), ' '.join(sys.argv)))
    log.info("config:\n{}".format(conf_pb))
    # Only thing we honor when running custom salt state is file lock
    # to avoid clashing with background salt. Other restrictions don't matter -
    # this is interactive debug instrument.
    f = runutil.Flag()
    if not manual:
        disabled, exists = f.is_enforced()
        if disabled:
            return False
        if exists:
            err = f.remove()
            if err is not None:
                log.error('Failed to remove disabled flag: {}'.format(err))
    if args.random_sleep:
        t = random.randint(1, args.random_sleep)
        log.info("Random sleeping {} ... ".format(t))
        time.sleep(t)

    if initial_setup.InitialSetup().need_initial_setup():
        log.info('Fail fast mode is enabled (initial setup phase)')
        execution_policy = policy.FailFast()
    elif args.fail_fast:
        execution_policy = policy.FailFast()
    else:
        execution_policy = policy.FailPermissive()

    if manual:
        ctx = context.Ctx(policy=execution_policy)
    else:
        def ctx_callback():
            i_am_disabled, _ = f.is_enforced()
            if i_am_disabled:
                return True, 'hostmanager has been disabled'
            else:
                return False, ''

        ctx = context.CallbackCtx(ctx_callback, policy=execution_policy)
    if args.local:
        mode = shim.LocalMode(args.local)
    else:
        mode = None
    status, err = shim.run_hm(ctx, conf_pb, mode=mode)
    if err is not None:
        log.error('HM run failed with error: {}'.format(err))
        return False
    if manual:
        sys.stdout.write(output.pformat_run(status, short=True))
    return True
