"""
CMD ya-salt status
"""
import logging
import sys
from google.protobuf import json_format

from infra.ya_salt.lib import persist
from infra.ya_salt.lib import output

log = logging.getLogger('salt-status')


def add_command(action):
    p = action.add_parser('status', help='Get status')
    p.add_argument('-o', '--output',
                   choices=['short', 'long', 'json', 'yaml'],
                   default='short',
                   help='Output format (default: %(default)s)')
    p.set_defaults(handle=handler)


def handler(args):
    log.setLevel(level=logging.INFO)
    status, err = persist.load_status()
    # Force colorless output if piped to something like less
    if not sys.stdout.isatty():
        args.nocolor = True
    if err is not None:
        log.error('Failed to load status (maybe has not run yet?): %s', err)
        return False
    else:
        if args.output == 'long':
            buf = output.pformat_run(status, nocolor=args.nocolor, short=False)
        elif args.output == 'short':
            buf = output.pformat_run(status, nocolor=args.nocolor, short=True)
        elif args.output == 'json':
            buf = json_format.MessageToJson(status,
                                            including_default_value_fields=True,  # For .jq grep
                                            sort_keys=True,  # To be able to test
                                            )
        elif args.output == 'yaml':
            import yaml
            d = json_format.MessageToDict(status, including_default_value_fields=True)
            buf = yaml.safe_dump(d)
        else:
            sys.stderr.write('Unsupported output format: {}'.format(args.output))
            return False
        sys.stdout.write(buf)
        return True
